<?php
session_start();

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../login.html');
    exit();
}

include_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

// Handle search and filters (same as search-candidates.php)
$search_term = $_GET['search'] ?? '';
$status_filter = $_GET['status'] ?? '';
$source_filter = $_GET['source'] ?? '';
$user_filter = $_GET['user_filter'] ?? '';
$date_filter_type = $_GET['date_filter_type'] ?? 'single';
$lineup_date = $_GET['lineup_date'] ?? '';
$lineup_date_from = $_GET['lineup_date_from'] ?? '';
$lineup_date_to = $_GET['lineup_date_to'] ?? '';

// Build the WHERE clause
$where_conditions = [];
$params = [];

if (!empty($search_term)) {
    $where_conditions[] = "(c.name LIKE ? OR c.candidate_id LIKE ? OR c.phone LIKE ? OR c.email LIKE ?)";
    $search_param = "%$search_term%";
    $params = array_merge($params, [$search_param, $search_param, $search_param, $search_param]);
}

if (!empty($status_filter)) {
    $where_conditions[] = "c.status = ?";
    $params[] = $status_filter;
}

if (!empty($source_filter)) {
    $where_conditions[] = "c.source = ?";
    $params[] = $source_filter;
}

if (!empty($user_filter)) {
    $where_conditions[] = "c.added_by = ?";
    $params[] = $user_filter;
}

// Date filtering
if ($date_filter_type === 'single' && !empty($lineup_date)) {
    $where_conditions[] = "DATE(c.lineup_date) = ?";
    $params[] = $lineup_date;
} elseif ($date_filter_type === 'range' && !empty($lineup_date_from) && !empty($lineup_date_to)) {
    $where_conditions[] = "DATE(c.lineup_date) BETWEEN ? AND ?";
    $params[] = $lineup_date_from;
    $params[] = $lineup_date_to;
}

$where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

// Get candidates with user information
$query = "SELECT c.*, u.username as added_by_name 
          FROM candidates c 
          LEFT JOIN users u ON c.added_by = u.id 
          $where_clause 
          ORDER BY c.lineup_date DESC, c.added_date DESC";

$stmt = $db->prepare($query);
$stmt->execute($params);
$candidates = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Set headers for CSV download
header('Content-Type: text/csv');
header('Content-Disposition: attachment; filename="candidates_export_' . date('Y-m-d_H-i-s') . '.csv"');

// Create file pointer
$output = fopen('php://output', 'w');

// Add CSV headers
fputcsv($output, [
    'Candidate ID',
    'Name',
    'Phone',
    'Email',
    'Status',
    'Source',
    'Lineup Date',
    'Added Date',
    'Added By',
    'Experience',
    'Skills',
    'Current Company',
    'Current Salary',
    'Expected Salary',
    'Notice Period',
    'Location'
]);

// Add data rows
foreach ($candidates as $candidate) {
    fputcsv($output, [
        $candidate['candidate_id'],
        $candidate['name'],
        $candidate['phone'],
        $candidate['email'],
        ucfirst(str_replace('_', ' ', $candidate['status'])),
        $candidate['source'],
        date('d-M-Y', strtotime($candidate['lineup_date'])),
        date('d-M-Y', strtotime($candidate['added_date'])),
        $candidate['added_by_name'] ?? 'Unknown',
        $candidate['experience'] ?? '',
        $candidate['skills'] ?? '',
        $candidate['current_company'] ?? '',
        $candidate['current_salary'] ?? '',
        $candidate['expected_salary'] ?? '',
        $candidate['notice_period'] ?? '',
        $candidate['location'] ?? ''
    ]);
}

fclose($output);
exit();
?>
