<?php
session_start();

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    exit('Unauthorized');
}

include_once '../config/database.php';
include_once '../classes/Candidate.php';

try {
    $database = new Database();
    $db = $database->getConnection();
    $candidate = new Candidate($db);

    $candidate_id = $_GET['candidate_id'] ?? '';
    $candidate->candidate_id = $candidate_id;

    if (!$candidate->readOne()) {
        throw new Exception('Candidate not found');
    }

    // Get interview history
    $interview_history_query = "SELECT * FROM interviews WHERE candidate_id = ? ORDER BY interview_date DESC LIMIT 5";
    $stmt = $db->prepare($interview_history_query);
    $stmt->execute([$candidate->id]);
    $interviews = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get status history
    $status_history_query = "SELECT * FROM candidate_status_history WHERE candidate_id = ? ORDER BY changed_at DESC LIMIT 10";
    $stmt = $db->prepare($status_history_query);
    $stmt->execute([$candidate->id]);
    $status_history = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get current interview process state if exists
    $process_state_query = "SELECT * FROM interview_process_state WHERE candidate_id = ? ORDER BY created_at DESC LIMIT 1";
    $stmt = $db->prepare($process_state_query);
    $stmt->execute([$candidate->id]);
    $current_state = $stmt->fetch(PDO::FETCH_ASSOC);

} catch (Exception $e) {
    http_response_code(500);
    exit('Error: ' . $e->getMessage());
}
?>

<div class="interview-process-container p-4">
    <!-- Candidate Information Header -->
    <div class="candidate-header mb-4 p-3 bg-light rounded">
        <div class="row">
            <div class="col-md-8">
                <h5 class="mb-1">
                    <i class="fas fa-user me-2"></i>
                    <?php echo htmlspecialchars($candidate->name); ?>
                    <span class="badge bg-primary ms-2"><?php echo $candidate->candidate_id; ?></span>
                </h5>
                <div class="candidate-details">
                    <p class="mb-1">
                        <i class="fas fa-phone text-muted me-1"></i><?php echo $candidate->phone; ?>
                        <span class="mx-2">|</span>
                        <i class="fas fa-envelope text-muted me-1"></i><?php echo $candidate->email; ?>
                    </p>
                    <p class="mb-1">
                        <i class="fas fa-calendar text-muted me-1"></i>
                        Lineup Date: <?php echo date('d-M-Y', strtotime($candidate->lineup_date)); ?>
                    </p>
                </div>
            </div>
            <div class="col-md-4 text-end">
                <div class="current-status">
                    <small class="text-muted">Current Status</small><br>
                    <span class="badge bg-<?php 
                        echo $candidate->status === 'pending' ? 'warning' : 
                            ($candidate->status === 'selected' ? 'success' : 
                            ($candidate->status === 'rejected' ? 'danger' : 
                            ($candidate->status === 'hold' ? 'secondary' :
                            ($candidate->status === 'shortlisted' ? 'info' : 
                            ($candidate->status === 'joined' ? 'success' : 'primary'))))); 
                    ?>">
                        <?php echo strtoupper(str_replace('_', ' ', $candidate->status)); ?>
                    </span>
                </div>
            </div>
        </div>
    </div>

    <!-- Process Progress Indicator -->
    <?php if ($current_state): ?>
    <div class="progress-indicator mb-4 p-3 bg-info bg-opacity-10 rounded">
        <h6><i class="fas fa-info-circle me-2"></i>Process Status</h6>
        <p class="mb-1"><strong>Current Step:</strong> <?php echo ucfirst(str_replace('_', ' ', $current_state['current_step'])); ?></p>
        <p class="mb-1"><strong>Last Updated:</strong> <?php echo date('d-M-Y H:i', strtotime($current_state['updated_at'])); ?></p>
        <?php if ($current_state['notes']): ?>
        <p class="mb-0"><strong>Notes:</strong> <?php echo htmlspecialchars($current_state['notes']); ?></p>
        <?php endif; ?>
    </div>
    <?php endif; ?>

    <!-- Status History -->
    <?php if (!empty($status_history)): ?>
    <div class="status-history mb-4 p-3 bg-light rounded">
        <h6><i class="fas fa-history me-2"></i>Status History</h6>
        <div class="table-responsive">
            <table class="table table-sm table-striped">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Previous Status</th>
                        <th>New Status</th>
                        <th>Reason</th>
                        <th>Changed By</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach (array_slice($status_history, 0, 5) as $history): ?>
                    <tr>
                        <td><?php echo date('d-M-Y H:i', strtotime($history['changed_at'])); ?></td>
                        <td>
                            <span class="badge bg-secondary">
                                <?php echo ucfirst(str_replace('_', ' ', $history['previous_status'] ?? 'N/A')); ?>
                            </span>
                        </td>
                        <td>
                            <span class="badge bg-primary">
                                <?php echo ucfirst(str_replace('_', ' ', $history['new_status'] ?? 'N/A')); ?>
                            </span>
                        </td>
                        <td>
                            <small><?php echo htmlspecialchars(substr($history['reason'] ?? '', 0, 50)) . (strlen($history['reason'] ?? '') > 50 ? '...' : ''); ?></small>
                        </td>
                        <td>
                            <small><?php echo htmlspecialchars($history['changed_by_name'] ?? 'System'); ?></small>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    <hr>
    <?php endif; ?>

    <!-- Interview History -->
    <?php if (!empty($interviews)): ?>
    <div class="interview-history mb-4 p-3 bg-light rounded">
        <h6><i class="fas fa-clipboard-list me-2"></i>Interview History</h6>
        <div class="table-responsive">
            <table class="table table-sm table-striped">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Type</th>
                        <th>Result</th>
                        <th>Notes</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($interviews as $int): ?>
                    <tr>
                        <td><?php echo date('d-M-Y', strtotime($int['interview_date'])); ?></td>
                        <td>
                            <span class="badge bg-info">
                                <?php echo ucfirst(str_replace('_', ' ', $int['interview_type'] ?? 'N/A')); ?>
                            </span>
                        </td>
                        <td>
                            <span class="badge bg-<?php 
                                echo ($int['result'] ?? '') === 'selected' ? 'success' : 
                                    (($int['result'] ?? '') === 'rejected' ? 'danger' : 
                                    (($int['result'] ?? '') === 'shortlisted' ? 'info' : 
                                    (($int['result'] ?? '') === 'hold' ? 'secondary' : 'warning'))); 
                            ?>">
                                <?php echo ucfirst($int['result'] ?? 'N/A'); ?>
                            </span>
                        </td>
                        <td>
                            <small><?php echo htmlspecialchars(substr($int['notes'] ?? '', 0, 50)) . (strlen($int['notes'] ?? '') > 50 ? '...' : ''); ?></small>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    <hr>
    <?php endif; ?>

    <!-- Interview Process Form -->
    <form method="POST" action="process-interview.php" id="interviewProcessForm">
        <input type="hidden" name="candidate_id" value="<?php echo $candidate->id; ?>">
        <input type="hidden" name="candidate_code" value="<?php echo $candidate->candidate_id; ?>">
        <input type="hidden" name="process_action" id="process_action">
        <input type="hidden" name="final_status" id="final_status">
        <input type="hidden" name="process_interview" value="1">
        <input type="hidden" name="save_step" id="save_step" value="0">
        
        <!-- Step 1: Turn Up Status -->
        <div class="step-section active border rounded p-4 mb-3" id="step1">
            <div class="step-header mb-3">
                <span class="badge bg-primary me-2">1</span>
                <h6 class="d-inline"><i class="fas fa-user-check me-2"></i>Turn Up Status</h6>
            </div>
            <p class="text-muted mb-3">Did the candidate turn up for the scheduled interview?</p>
            
            <div class="row">
                <div class="col-md-6 mb-2">
                    <button type="button" class="btn btn-outline-success btn-lg w-100 turn-up-btn" onclick="handleTurnUp(true)">
                        <i class="fas fa-check-circle me-2"></i>
                        <div>
                            <strong>Yes - Turned Up</strong>
                            <small class="d-block">Candidate attended the interview</small>
                        </div>
                    </button>
                </div>
                <div class="col-md-6 mb-2">
                    <button type="button" class="btn btn-outline-danger btn-lg w-100 turn-up-btn" onclick="handleTurnUp(false)">
                        <i class="fas fa-times-circle me-2"></i>
                        <div>
                            <strong>No - Didn't Turn Up</strong>
                            <small class="d-block">Candidate was absent</small>
                        </div>
                    </button>
                </div>
            </div>
            
            <!-- Save and Continue Button -->
            <div class="text-center mt-4" id="step1_save" style="display: none;">
                <button type="button" class="btn btn-primary btn-lg" onclick="saveAndContinue(1)">
                    <i class="fas fa-save me-2"></i>Save & Continue to Next Step
                </button>
            </div>
        </div>

        <!-- Step 2A: If No - Reschedule Options -->
        <div class="step-section border rounded p-4 mb-3" id="step2a" style="display: none;">
            <div class="step-header mb-3">
                <span class="badge bg-warning me-2">2</span>
                <h6 class="d-inline"><i class="fas fa-exclamation-triangle me-2"></i>Candidate Didn't Turn Up</h6>
            </div>
            
            <div class="alert alert-warning">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <strong>Candidate didn't turn up.</strong> What would you like to do?
            </div>
            
            <div class="row">
                <div class="col-md-6 mb-3">
                    <div class="card h-100 action-card" onclick="showRescheduleForm()">
                        <div class="card-body text-center">
                            <i class="fas fa-calendar-plus fa-3x text-primary mb-3"></i>
                            <h5 class="card-title">Reschedule Interview</h5>
                            <p class="card-text text-muted">Set a new lineup date for the candidate</p>
                            <button type="button" class="btn btn-primary">
                                <i class="fas fa-calendar-plus me-1"></i>Reschedule
                            </button>
                        </div>
                    </div>
                </div>
                <div class="col-md-6 mb-3">
                    <div class="card h-100 action-card" onclick="showEndProcessForm()">
                        <div class="card-body text-center">
                            <i class="fas fa-times-circle fa-3x text-danger mb-3"></i>
                            <h5 class="card-title">End Process</h5>
                            <p class="card-text text-muted">Close the interview process permanently</p>
                            <button type="button" class="btn btn-danger">
                                <i class="fas fa-times me-1"></i>End Process
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Step 2A1: Reschedule Form -->
        <div class="step-section border rounded p-4 mb-3" id="step2a1" style="display: none;">
            <div class="step-header mb-3">
                <span class="badge bg-primary me-2">2</span>
                <h6 class="d-inline"><i class="fas fa-calendar-plus me-2"></i>Reschedule Interview</h6>
            </div>
            
            <div class="alert alert-info">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Reschedule Interview:</strong> Set a new lineup date for this candidate.
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="next_lineup_date" class="form-label">Next Lineup Date <span class="text-danger">*</span></label>
                    <input type="date" class="form-control" name="next_lineup_date" id="next_lineup_date" 
                           min="<?php echo date('Y-m-d', strtotime('+1 day')); ?>" required>
                </div>
                <div class="col-md-6">
                    <label for="not_coming_reason" class="form-label">Reason for Not Coming <span class="text-danger">*</span></label>
                    <select class="form-select" name="not_coming_reason" id="not_coming_reason" required>
                        <option value="">Select Reason</option>
                        <option value="Personal Emergency">Personal Emergency</option>
                        <option value="Health Issues">Health Issues</option>
                        <option value="Transport Issues">Transport Issues</option>
                        <option value="Other Commitment">Other Commitment</option>
                        <option value="Changed Mind">Changed Mind</option>
                        <option value="Family Issues">Family Issues</option>
                        <option value="Weather Issues">Weather Issues</option>
                        <option value="No Response">No Response from Candidate</option>
                        <option value="Other">Other</option>
                    </select>
                </div>
            </div>
            
            <div class="mb-4">
                <label for="reason_details" class="form-label">Additional Details (Optional)</label>
                <textarea class="form-control" name="reason_details" id="reason_details" rows="3" 
                          placeholder="Any additional information about the absence..."></textarea>
            </div>
            
            <div class="text-center">
                <button type="button" class="btn btn-secondary me-2" onclick="goBackToOptions()">
                    <i class="fas fa-arrow-left me-1"></i>Back
                </button>
                <button type="button" class="btn btn-warning me-2" onclick="saveAndContinue('2a1')">
                    <i class="fas fa-save me-1"></i>Save Progress
                </button>
                <button type="button" class="btn btn-primary btn-lg" onclick="rescheduleCandidate()">
                    <i class="fas fa-calendar-plus me-2"></i>Reschedule Interview
                </button>
            </div>
        </div>

        <!-- Step 2A2: End Process Form -->
        <div class="step-section border rounded p-4 mb-3" id="step2a2" style="display: none;">
            <div class="step-header mb-3">
                <span class="badge bg-danger me-2">2</span>
                <h6 class="d-inline"><i class="fas fa-times-circle me-2"></i>End Interview Process</h6>
            </div>
            
            <div class="alert alert-danger">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <strong>Warning:</strong> This will permanently end the interview process for this candidate.
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="end_reason" class="form-label">Reason for Ending Process <span class="text-danger">*</span></label>
                    <select class="form-select" name="not_coming_reason" id="end_reason" required>
                        <option value="">Select Reason</option>
                        <option value="No Show Multiple Times">No Show Multiple Times</option>
                        <option value="Candidate Not Interested">Candidate Not Interested</option>
                        <option value="Poor Communication">Poor Communication</option>
                        <option value="Not Meeting Requirements">Not Meeting Requirements</option>
                        <option value="Unprofessional Behavior">Unprofessional Behavior</option>
                        <option value="Other">Other</option>
                    </select>
                </div>
            </div>
            
            <div class="mb-4">
                <label for="end_details" class="form-label">Additional Details (Optional)</label>
                <textarea class="form-control" name="reason_details" id="end_details" rows="3" 
                          placeholder="Any additional information about why the process is being ended..."></textarea>
            </div>
            
            <div class="text-center">
                <button type="button" class="btn btn-secondary me-2" onclick="goBackToOptions()">
                    <i class="fas fa-arrow-left me-1"></i>Back
                </button>
                <button type="button" class="btn btn-warning me-2" onclick="saveAndContinue('2a2')">
                    <i class="fas fa-save me-1"></i>Save Progress
                </button>
                <button type="button" class="btn btn-danger btn-lg" onclick="endProcess()">
                    <i class="fas fa-times me-2"></i>End Process
                </button>
            </div>
        </div>

        <!-- Step 2B: If Yes - Interview Result Selection -->
        <div class="step-section border rounded p-4 mb-3" id="step2b" style="display: none;">
            <div class="step-header mb-3">
                <span class="badge bg-success me-2">2</span>
                <h6 class="d-inline"><i class="fas fa-clipboard-list me-2"></i>Interview Result</h6>
            </div>
            
            <div class="alert alert-success">
                <i class="fas fa-check-circle me-2"></i>
                <strong>Great!</strong> Candidate attended the interview. Please select the interview result:
            </div>
            
            <!-- Interview Basic Details -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="interview_type" class="form-label">Interview Type <span class="text-danger">*</span></label>
                    <select class="form-select" name="interview_type" id="interview_type" required>
                        <option value="">Select Interview Type</option>
                        <option value="initial">Initial Screening</option>
                        <option value="technical">Technical Interview</option>
                        <option value="hr">HR Interview</option>
                        <option value="final">Final Interview</option>
                        <option value="group_discussion">Group Discussion</option>
                        <option value="practical_test">Practical Test</option>
                    </select>
                </div>
                <div class="col-md-6">
                    <label for="interviewer_name" class="form-label">Interviewer Name <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" name="interviewer_name" id="interviewer_name" 
                           value="<?php echo htmlspecialchars($_SESSION['username'] ?? ''); ?>" required>
                </div>
            </div>
            
            <div class="mb-4">
                <label for="interview_notes" class="form-label">Interview Notes & Feedback</label>
                <textarea class="form-control" name="interview_notes" id="interview_notes" rows="4" 
                          placeholder="Interview feedback, observations, technical skills assessment, communication skills, overall impression..."></textarea>
            </div>

            <!-- Interview Result Options -->
            <div class="mb-4">
                <label for="interview_result" class="form-label">Interview Result <span class="text-danger">*</span></label>
                <select class="form-select form-select-lg" name="interview_result" id="interview_result" required onchange="handleInterviewResult()">
                    <option value="">Select Interview Result</option>
                    <option value="selected">
                        <i class="fas fa-trophy"></i> SELECTED - Candidate is selected for the role
                    </option>
                    <option value="shortlisted">
                        <i class="fas fa-list"></i> SHORTLISTED - Move to next interview round
                    </option>
                    <option value="hold">
                        <i class="fas fa-pause"></i> HOLD - Decision pending
                    </option>
                    <option value="rejected">
                        <i class="fas fa-times"></i> REJECTED - Not suitable for the role
                    </option>
                </select>
                <div class="form-text">Please select the interview result to proceed with the next steps.</div>
            </div>
            
            <!-- Save and Continue Button -->
            <div class="text-center" id="step2b_save" style="display: none;">
                <button type="button" class="btn btn-warning me-2" onclick="saveAndContinue('2b')">
                    <i class="fas fa-save me-1"></i>Save Progress
                </button>
                <button type="button" class="btn btn-primary btn-lg" onclick="proceedToNextStep()">
                    <i class="fas fa-arrow-right me-2"></i>Continue to Next Step
                </button>
            </div>
        </div>

        <!-- Step 3A: If Rejected - Reason and End Process -->
        <div class="step-section border rounded p-4 mb-3" id="step3a" style="display: none;">
            <div class="step-header mb-3">
                <span class="badge bg-danger me-2">3</span>
                <h6 class="d-inline"><i class="fas fa-times-circle me-2"></i>Rejection Details</h6>
            </div>
            
            <div class="alert alert-danger">
                <i class="fas fa-times-circle me-2"></i>
                <strong>Candidate Rejected!</strong> Please provide rejection reason.
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="rejection_reason" class="form-label">Rejection Reason <span class="text-danger">*</span></label>
                    <select class="form-select" name="rejection_reason" id="rejection_reason" required>
                        <option value="">Select Reason</option>
                        <option value="Technical Skills Gap">Technical Skills Gap</option>
                        <option value="Communication Issues">Communication Issues</option>
                        <option value="Experience Not Matching">Experience Not Matching</option>
                        <option value="Cultural Fit Issues">Cultural Fit Issues</option>
                        <option value="Salary Expectations Too High">Salary Expectations Too High</option>
                        <option value="Attitude Issues">Attitude Issues</option>
                        <option value="Failed Technical Test">Failed Technical Test</option>
                        <option value="Other">Other</option>
                    </select>
                </div>
            </div>
            
            <div class="mb-4">
                <label for="rejection_details" class="form-label">Additional Details</label>
                <textarea class="form-control" name="rejection_details" id="rejection_details" rows="3" 
                          placeholder="Additional details about the rejection..."></textarea>
            </div>
            
            <div class="text-center">
                <button type="button" class="btn btn-warning me-2" onclick="saveAndContinue('3a')">
                    <i class="fas fa-save me-1"></i>Save Progress
                </button>
                <button type="button" class="btn btn-danger btn-lg" onclick="confirmRejection()">
                    <i class="fas fa-times me-2"></i>Confirm Rejection & End Process
                </button>
            </div>
        </div>

        <!-- Step 3B: If Hold - Reason and End Process -->
        <div class="step-section border rounded p-4 mb-3" id="step3b" style="display: none;">
            <div class="step-header mb-3">
                <span class="badge bg-secondary me-2">3</span>
                <h6 class="d-inline"><i class="fas fa-pause-circle me-2"></i>Hold Details</h6>
            </div>
            
            <div class="alert alert-warning">
                <i class="fas fa-pause-circle me-2"></i>
                <strong>Candidate On Hold!</strong> Please provide reason for hold.
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="hold_reason" class="form-label">Hold Reason <span class="text-danger">*</span></label>
                    <select class="form-select" name="hold_reason" id="hold_reason" required>
                        <option value="">Select Reason</option>
                        <option value="Need Management Approval">Need Management Approval</option>
                        <option value="Budget Constraints">Budget Constraints</option>
                        <option value="Position On Hold">Position On Hold</option>
                        <option value="Comparing with Other Candidates">Comparing with Other Candidates</option>
                        <option value="Reference Check Pending">Reference Check Pending</option>
                        <option value="Other">Other</option>
                    </select>
                </div>
            </div>
            
            <div class="mb-4">
                <label for="hold_details" class="form-label">Additional Details</label>
                <textarea class="form-control" name="hold_details" id="hold_details" rows="3" 
                          placeholder="Additional details about putting candidate on hold..."></textarea>
            </div>
            
            <div class="text-center">
                <button type="button" class="btn btn-warning me-2" onclick="saveAndContinue('3b')">
                    <i class="fas fa-save me-1"></i>Save Progress
                </button>
                <button type="button" class="btn btn-warning btn-lg" onclick="confirmHold()">
                    <i class="fas fa-pause me-2"></i>Confirm Hold & End Process
                </button>
            </div>
        </div>

        <!-- Step 3C: If Shortlisted - Turn Up for Next Round -->
        <div class="step-section border rounded p-4 mb-3" id="step3c" style="display: none;">
            <div class="step-header mb-3">
                <span class="badge bg-info me-2">3</span>
                <h6 class="d-inline"><i class="fas fa-calendar-check me-2"></i>Next Interview Round</h6>
            </div>
            
            <div class="alert alert-info">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Candidate Shortlisted!</strong> Schedule next interview and check turn up status.
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="next_interview_date" class="form-label">Next Interview Date <span class="text-danger">*</span></label>
                    <input type="date" class="form-control" name="next_interview_date" id="next_interview_date" 
                           min="<?php echo date('Y-m-d', strtotime('+1 day')); ?>" required>
                </div>
                <div class="col-md-6">
                    <label for="next_interview_type" class="form-label">Next Interview Type <span class="text-danger">*</span></label>
                    <select class="form-select" name="next_interview_type" id="next_interview_type" required>
                        <option value="">Select Next Interview Type</option>
                        <option value="technical">Technical Interview</option>
                        <option value="hr">HR Interview</option>
                        <option value="final">Final Interview</option>
                        <option value="group_discussion">Group Discussion</option>
                        <option value="practical_test">Practical Test</option>
                        <option value="management">Management Round</option>
                    </select>
                </div>
            </div>
            
            <div class="mb-4">
                <label for="next_interview_notes" class="form-label">Instructions for Next Interview</label>
                <textarea class="form-control" name="next_interview_notes" id="next_interview_notes" rows="3" 
                          placeholder="Special instructions, topics to focus on, skills to assess..."></textarea>
            </div>
            
            <div class="text-center mb-4">
                <button type="button" class="btn btn-warning me-2" onclick="saveAndContinue('3c')">
                    <i class="fas fa-save me-1"></i>Save Progress
                </button>
                <button type="button" class="btn btn-info btn-lg" onclick="scheduleNextInterview()">
                    <i class="fas fa-calendar-plus me-2"></i>Schedule Next Interview
                </button>
            </div>
            
            <!-- Turn Up Status for Next Round -->
            <div class="border-top pt-4" id="next_round_turnup" style="display: none;">
                <h6 class="mb-3"><i class="fas fa-user-check me-2"></i>Next Round Turn Up Status</h6>
                <p class="text-muted mb-3">Did the candidate turn up for the next interview round?</p>
                
                <div class="row">
                    <div class="col-md-6 mb-2">
                        <button type="button" class="btn btn-outline-success btn-lg w-100" onclick="handleNextRoundTurnUp(true)">
                            <i class="fas fa-check-circle me-2"></i>
                            <div>
                                <strong>Yes - Turned Up</strong>
                                <small class="d-block">Continue with interview process</small>
                            </div>
                        </button>
                    </div>
                    <div class="col-md-6 mb-2">
                        <button type="button" class="btn btn-outline-danger btn-lg w-100" onclick="handleNextRoundTurnUp(false)">
                            <i class="fas fa-times-circle me-2"></i>
                            <div>
                                <strong>No - Didn't Turn Up</strong>
                                <small class="d-block">Handle absence</small>
                            </div>
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Step 3D: If Selected - Offer Rollout -->
        <div class="step-section border rounded p-4 mb-3" id="step3d" style="display: none;">
            <div class="step-header mb-3">
                <span class="badge bg-success me-2">3</span>
                <h6 class="d-inline"><i class="fas fa-trophy me-2"></i>Offer Rollout</h6>
            </div>
            
            <div class="alert alert-success">
                <i class="fas fa-trophy me-2"></i>
                <strong>Congratulations!</strong> Candidate has been selected. Proceed with offer rollout?
            </div>
            
            <div class="row">
                <div class="col-md-6 mb-2">
                    <button type="button" class="btn btn-outline-success btn-lg w-100" onclick="proceedWithOffer()">
                        <i class="fas fa-handshake me-2"></i>
                        <div>
                            <strong>Yes - Rollout Offer</strong>
                            <small class="d-block">Proceed with offer process</small>
                        </div>
                    </button>
                </div>
                <div class="col-md-6 mb-2">
                    <button type="button" class="btn btn-outline-warning btn-lg w-100" onclick="holdOffer()">
                        <i class="fas fa-clock me-2"></i>
                        <div>
                            <strong>No - Hold Offer</strong>
                            <small class="d-block">Keep selected but hold offer</small>
                        </div>
                    </button>
                </div>
            </div>
            
            <!-- Save Progress Button -->
            <div class="text-center mt-3">
                <button type="button" class="btn btn-warning" onclick="saveAndContinue('3d')">
                    <i class="fas fa-save me-1"></i>Save Progress
                </button>
            </div>
        </div>

        <!-- Step 4: Offer Acceptance -->
        <div class="step-section border rounded p-4 mb-3" id="step4" style="display: none;">
            <div class="step-header mb-3">
                <span class="badge bg-success me-2">4</span>
                <h6 class="d-inline"><i class="fas fa-handshake me-2"></i>Offer Acceptance</h6>
            </div>
            
            <div class="alert alert-info">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Offer Rolled Out!</strong> Did the candidate accept the offer?
            </div>
            
            <div class="row">
                <div class="col-md-6 mb-2">
                    <button type="button" class="btn btn-outline-success btn-lg w-100" onclick="handleOfferAcceptance(true)">
                        <i class="fas fa-check-circle me-2"></i>
                        <div>
                            <strong>Yes - Offer Accepted</strong>
                            <small class="d-block">Proceed to joining details</small>
                        </div>
                    </button>
                </div>
                <div class="col-md-6 mb-2">
                    <button type="button" class="btn btn-outline-danger btn-lg w-100" onclick="handleOfferAcceptance(false)">
                        <i class="fas fa-times-circle me-2"></i>
                        <div>
                            <strong>No - Offer Declined</strong>
                            <small class="d-block">End process</small>
                        </div>
                    </button>
                </div>
            </div>
            
            <!-- Save Progress Button -->
            <div class="text-center mt-3">
                <button type="button" class="btn btn-warning" onclick="saveAndContinue('4')">
                    <i class="fas fa-save me-1"></i>Save Progress
                </button>
            </div>
        </div>

        <!-- Step 5: Joining Details -->
        <div class="step-section border rounded p-4 mb-3" id="step5" style="display: none;">
            <div class="step-header mb-3">
                <span class="badge bg-success me-2">5</span>
                <h6 class="d-inline"><i class="fas fa-calendar-check me-2"></i>Joining Details</h6>
            </div>
            
            <div class="alert alert-success">
                <i class="fas fa-handshake me-2"></i>
                <strong>Offer Accepted!</strong> Please enter the joining details.
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="joining_date" class="form-label">Expected Joining Date <span class="text-danger">*</span></label>
                    <input type="date" class="form-control" name="joining_date" id="joining_date" 
                           min="<?php echo date('Y-m-d'); ?>" required>
                </div>
                <div class="col-md-6">
                    <label for="salary_offered" class="form-label">Salary Offered (Annual)</label>
                    <input type="number" class="form-control" name="salary_offered" id="salary_offered" 
                           placeholder="e.g., 500000" min="0">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="position" class="form-label">Position/Designation</label>
                    <input type="text" class="form-control" name="position" id="position" 
                           placeholder="e.g., Software Developer">
                </div>
                <div class="col-md-6">
                    <label for="department" class="form-label">Department</label>
                    <input type="text" class="form-control" name="department" id="department" 
                           placeholder="e.g., IT Department">
                </div>
            </div>
            
            <div class="mb-4">
                <label for="joining_notes" class="form-label">Joining Instructions</label>
                <textarea class="form-control" name="joining_notes" id="joining_notes" rows="3" 
                          placeholder="Documents required, reporting details, first day instructions..."></textarea>
            </div>
            
            <div class="text-center">
                <button type="button" class="btn btn-warning me-2" onclick="saveAndContinue('5')">
                    <i class="fas fa-save me-1"></i>Save Progress
                </button>
                <button type="button" class="btn btn-success btn-lg" onclick="setJoiningDetails()">
                    <i class="fas fa-calendar-check me-2"></i>Set Joining Details
                </button>
            </div>
        </div>

        <!-- Step 6: Joining Confirmation -->
        <div class="step-section border rounded p-4 mb-3" id="step6" style="display: none;">
            <div class="step-header mb-3">
                <span class="badge bg-primary me-2">6</span>
                <h6 class="d-inline"><i class="fas fa-user-plus me-2"></i>Joining Confirmation</h6>
            </div>
            
            <div class="alert alert-info">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Joining Details Set!</strong> Did the candidate join on the scheduled date?
            </div>
            
            <div class="mb-3">
                <label for="actual_joining_date" class="form-label">Actual Joining Date</label>
                <input type="date" class="form-control" name="actual_joining_date" id="actual_joining_date" 
                       max="<?php echo date('Y-m-d'); ?>">
                <div class="form-text">Leave empty if candidate didn't join</div>
            </div>
            
            <div class="mb-4">
                <label for="joining_feedback" class="form-label">Joining Feedback/Notes</label>
                <textarea class="form-control" name="joining_feedback" id="joining_feedback" rows="3" 
                          placeholder="Any feedback about the joining process, first day experience, or reasons for not joining..."></textarea>
            </div>
            
            <div class="row">
                <div class="col-md-6 mb-2">
                    <button type="button" class="btn btn-outline-success btn-lg w-100" onclick="handleJoiningConfirmation(true)">
                        <i class="fas fa-check-circle me-2"></i>
                        <div>
                            <strong>Yes - Candidate Joined</strong>
                            <small class="d-block">Successfully joined the organization</small>
                        </div>
                    </button>
                </div>
                <div class="col-md-6 mb-2">
                    <button type="button" class="btn btn-outline-danger btn-lg w-100" onclick="handleJoiningConfirmation(false)">
                        <i class="fas fa-times-circle me-2"></i>
                        <div>
                            <strong>No - Candidate Didn't Join</strong>
                            <small class="d-block">Did not join as expected</small>
                        </div>
                    </button>
                </div>
            </div>
            
            <!-- Save Progress Button -->
            <div class="text-center mt-3">
                <button type="button" class="btn btn-warning" onclick="saveAndContinue('6')">
                    <i class="fas fa-save me-1"></i>Save Progress
                </button>
            </div>
        </div>

        <!-- Process Complete -->
        <div class="step-section border rounded p-4 mb-3 bg-light" id="process_complete" style="display: none;">
            <div class="text-center">
                <i class="fas fa-check-circle text-success" style="font-size: 3rem;"></i>
                <h4 class="mt-3 text-success">Process Complete!</h4>
                <p class="text-muted" id="completion_message">The interview process has been completed successfully.</p>
                <button type="button" class="btn btn-primary" data-bs-dismiss="modal">
                    <i class="fas fa-times me-2"></i>Close
                </button>
            </div>
        </div>
    </form>

    <!-- Action Buttons -->
    <div class="text-center mt-4 pt-3 border-top" id="action_buttons">
        <button type="button" class="btn btn-secondary btn-lg" data-bs-dismiss="modal">
            <i class="fas fa-times me-2"></i>Cancel
        </button>
    </div>
</div>

<style>
.step-section {
    transition: all 0.3s ease;
}
.step-section.active {
    border-color: #0d6efd !important;
    box-shadow: 0 0 0 0.2rem rgba(13, 110, 253, 0.25);
}
.turn-up-btn:hover, .result-btn:hover, .joining-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
}
.step-header {
    border-bottom: 1px solid #dee2e6;
    padding-bottom: 10px;
}
.action-card {
    cursor: pointer;
    transition: all 0.3s ease;
    border: 2px solid #dee2e6;
}
.action-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.15);
    border-color: #007bff;
}
.action-card .card-body {
    padding: 2rem;
}
.action-card .btn {
    pointer-events: none;
}

/* Custom styling for select options */
#interview_result option {
    padding: 10px;
    font-weight: 500;
}

#interview_result option[value="selected"] {
    background-color: #d1ecf1;
    color: #0c5460;
}

#interview_result option[value="shortlisted"] {
    background-color: #d4edda;
    color: #155724;
}

#interview_result option[value="hold"] {
    background-color: #fff3cd;
    color: #856404;
}

#interview_result option[value="rejected"] {
    background-color: #f8d7da;
    color: #721c24;
}

.progress-indicator {
    border-left: 4px solid #17a2b8;
}

.btn.loading {
    position: relative;
    color: transparent;
}

.btn.loading::after {
    content: "";
    position: absolute;
    width: 16px;
    height: 16px;
    top: 50%;
    left: 50%;
    margin-left: -8px;
    margin-top: -8px;
    border: 2px solid transparent;
    border-top-color: #ffffff;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}
</style>

<script>
let currentStep = 1;
let selectedResult = '';

function handleTurnUp(turnedUp) {
    // Remove active class from all steps except step1
    document.querySelectorAll('.step-section:not(#step1)').forEach(step => {
        step.classList.remove('active');
        step.style.display = 'none';
    });
    
    // Highlight selected button
    document.querySelectorAll('.turn-up-btn').forEach(btn => {
        btn.classList.remove('selected');
    });
    
    // Find the clicked button properly
    const clickedButton = event.currentTarget;
    clickedButton.classList.add('selected');
    
    // Show save button for step 1
    document.getElementById('step1_save').style.display = 'block';
    
    // Store the turn up status
    window.turnUpStatus = turnedUp;
}

function saveAndContinue(step) {
    // Show loading state
    const saveBtn = event.currentTarget;
    saveBtn.classList.add('loading');
    saveBtn.disabled = true;
    
    // Collect form data
    const formData = new FormData(document.getElementById('interviewProcessForm'));
    formData.append('save_step', '1');
    formData.append('current_step', step);
    
    // Add step-specific data
    if (step === 1) {
        formData.append('turn_up_status', window.turnUpStatus ? '1' : '0');
    }
    
    // Save progress via AJAX
    fetch('ajax/save-interview-progress.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Show success message
            showNotification('Progress saved successfully!', 'success');
            
            // Continue to next step based on current step
            if (step === 1) {
                if (window.turnUpStatus) {
                    showStep('step2b');
                } else {
                    showStep('step2a');
                }
            } else if (step === '2b') {
                proceedToNextStep();
            } else {
                // For other steps, just show success and stay on current step
                setTimeout(() => {
                    saveBtn.classList.remove('loading');
                    saveBtn.disabled = false;
                }, 1000);
            }
        } else {
            showNotification('Error saving progress: ' + data.message, 'error');
            saveBtn.classList.remove('loading');
            saveBtn.disabled = false;
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Error saving progress', 'error');
        saveBtn.classList.remove('loading');
        saveBtn.disabled = false;
    });
}

function showStep(stepId) {
    // Hide all steps
    document.querySelectorAll('.step-section').forEach(step => {
        step.classList.remove('active');
        step.style.display = 'none';
    });
    
    // Show target step
    const targetStep = document.getElementById(stepId);
    if (targetStep) {
        targetStep.style.display = 'block';
        targetStep.classList.add('active');
        
        setTimeout(() => {
            targetStep.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }, 100);
    }
}

function showNotification(message, type) {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `alert alert-${type === 'success' ? 'success' : 'danger'} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-circle'} me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(notification);
    
    // Auto remove after 3 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 3000);
}

function handleInterviewResult() {
    const result = document.getElementById('interview_result').value;
    const interviewType = document.getElementById('interview_type').value;
    const interviewerName = document.getElementById('interviewer_name').value;
    
    if (!interviewType) {
        alert('Please select interview type first');
        document.getElementById('interview_type').focus();
        document.getElementById('interview_result').value = '';
        return;
    }
    
    if (!interviewerName.trim()) {
        alert('Please enter interviewer name');
        document.getElementById('interviewer_name').focus();
        document.getElementById('interview_result').value = '';
        return;
    }
    
    if (!result) {
        return;
    }
    
    selectedResult = result;
    
    // Show save button for step 2b
    document.getElementById('step2b_save').style.display = 'block';
    
    // Set process action and final status
    document.getElementById('process_action').value = result;
    document.getElementById('final_status').value = result;
}

function proceedToNextStep() {
    if (!selectedResult) {
        alert('Please select interview result first');
        return;
    }
    
    // Hide current step
    document.getElementById('step2b').classList.remove('active');
    
    // Show appropriate next step based on result
    switch(selectedResult) {
        case 'rejected':
            showStep('step3a');
            break;
        case 'hold':
            showStep('step3b');
            break;
        case 'shortlisted':
            showStep('step3c');
            break;
        case 'selected':
            showStep('step3d');
            break;
    }
    
    currentStep = 3;
}

function confirmRejection() {
    const reason = document.getElementById('rejection_reason').value;
    
    if (!reason) {
        alert('Please select rejection reason');
        document.getElementById('rejection_reason').focus();
        return;
    }
    
    if (confirm('Are you sure you want to reject this candidate? This will end the interview process.')) {
        document.getElementById('process_action').value = 'reject';
        document.getElementById('final_status').value = 'rejected';
        submitInterviewProcess();
    }
}

function confirmHold() {
    const reason = document.getElementById('hold_reason').value;
    
    if (!reason) {
        alert('Please select hold reason');
        document.getElementById('hold_reason').focus();
        return;
    }
    
    if (confirm('Are you sure you want to put this candidate on hold? This will end the current interview process.')) {
        document.getElementById('process_action').value = 'hold';
        document.getElementById('final_status').value = 'hold';
        submitInterviewProcess();
    }
}

function scheduleNextInterview() {
    const nextDate = document.getElementById('next_interview_date').value;
    const nextType = document.getElementById('next_interview_type').value;
    
    if (!nextDate) {
        alert('Please select next interview date');
        document.getElementById('next_interview_date').focus();
        return;
    }
    
    if (!nextType) {
        alert('Please select next interview type');
        document.getElementById('next_interview_type').focus();
        return;
    }
    
    // Show confirmation message
    if (confirm(`Schedule next ${nextType} interview for ${nextDate}?`)) {
        // Show turn up status for next round immediately
        document.getElementById('next_round_turnup').style.display = 'block';
        
        // Update the turn up section header to show the scheduled interview details
        const turnUpHeader = document.querySelector('#next_round_turnup h6');
        turnUpHeader.innerHTML = `
            <i class="fas fa-user-check me-2"></i>
            Next Round Turn Up Status - ${nextType.charAt(0).toUpperCase() + nextType.slice(1).replace('_', ' ')} Interview (${new Date(nextDate).toLocaleDateString()})
        `;
        
        setTimeout(() => {
            document.getElementById('next_round_turnup').scrollIntoView({ behavior: 'smooth', block: 'center' });
        }, 100);
    }
}

function handleNextRoundTurnUp(turnedUp) {
    if (turnedUp) {
        // If turned up, go back to step 2b for next round interview result
        showStep('step2b');
        
        // Clear previous selections to start fresh for this round
        document.getElementById('interview_result').value = '';
        document.getElementById('interview_notes').value = '';
        
        // Update the alert message to indicate this is a continuation
        const alertDiv = document.querySelector('#step2b .alert-success');
        alertDiv.innerHTML = `
            <i class="fas fa-check-circle me-2"></i>
            <strong>Great!</strong> Candidate turned up for the next interview round. Please select the interview result for this round:
        `;
    } else {
        // If didn't turn up, handle as reschedule or end process
        if (confirm('Candidate didn\'t turn up for the next round. Would you like to reschedule or end the process?')) {
            showStep('step2a');
        }
    }
}

function proceedWithOffer() {
    showStep('step4');
    currentStep = 4;
}

function holdOffer() {
    if (confirm('Hold the offer for this candidate? Status will be set to selected but offer will be on hold.')) {
        document.getElementById('process_action').value = 'select_hold_offer';
        document.getElementById('final_status').value = 'selected';
        submitInterviewProcess();
    }
}

function handleOfferAcceptance(accepted) {
    if (accepted) {
        showStep('step5');
        currentStep = 5;
    } else {
        if (confirm('Candidate declined the offer. This will end the process with status "offer_declined".')) {
            document.getElementById('process_action').value = 'offer_declined';
            document.getElementById('final_status').value = 'offer_declined';
            submitInterviewProcess();
        }
    }
}

function setJoiningDetails() {
    const joiningDate = document.getElementById('joining_date').value;
    
    if (!joiningDate) {
        alert('Please enter expected joining date');
        document.getElementById('joining_date').focus();
        return;
    }
    
    // Show joining confirmation step instead of completing process
    showStep('step6');
    currentStep = 6;
    
    // Set the actual joining date to the expected joining date by default
    document.getElementById('actual_joining_date').value = joiningDate;
}

function handleJoiningConfirmation(joined) {
    const actualJoiningDate = document.getElementById('actual_joining_date').value;
    const joiningFeedback = document.getElementById('joining_feedback').value;
    
    if (joined && !actualJoiningDate) {
        alert('Please enter actual joining date');
        document.getElementById('actual_joining_date').focus();
        return;
    }
    
    let confirmMessage = joined ? 
        'Confirm that the candidate has joined successfully?' : 
        'Confirm that the candidate did not join?';
        
    if (confirm(confirmMessage)) {
        if (joined) {
            document.getElementById('process_action').value = 'joined';
            document.getElementById('final_status').value = 'joined';
        } else {
            document.getElementById('process_action').value = 'not_joined';
            document.getElementById('final_status').value = 'not_joined';
        }
        
        // Show completion message
        showStep('process_complete');
        
        // Update completion message based on joining status
        const completionMessage = document.getElementById('completion_message');
        if (joined) {
            completionMessage.innerHTML = 'The interview process has been completed successfully. Candidate has joined the organization!';
        } else {
            completionMessage.innerHTML = 'The interview process has been completed. Candidate was selected but did not join.';
        }
        
        // Submit after showing completion
        setTimeout(() => {
            submitInterviewProcess();
        }, 2000);
    }
}

// Reschedule and End Process functions
function showRescheduleForm() {
    showStep('step2a1');
}

function showEndProcessForm() {
    showStep('step2a2');
}

function goBackToOptions() {
    showStep('step2a');
}

function rescheduleCandidate() {
    const nextDate = document.getElementById('next_lineup_date').value;
    const reason = document.getElementById('not_coming_reason').value;
    
    if (!nextDate) {
        alert('Please select next lineup date');
        document.getElementById('next_lineup_date').focus();
        return;
    }
    
    if (!reason) {
        alert('Please select reason for not coming');
        document.getElementById('not_coming_reason').focus();
        return;
    }
    
    if (confirm('Reschedule this candidate for ' + nextDate + '?')) {
        document.getElementById('process_action').value = 'reschedule';
        document.getElementById('final_status').value = 'rescheduled';
        submitInterviewProcess();
    }
}

function endProcess() {
    const reason = document.getElementById('end_reason').value;
    
    if (!reason) {
        alert('Please select reason for ending process');
        document.getElementById('end_reason').focus();
        return;
    }
    
    if (confirm('Are you sure you want to end the interview process for this candidate? This action cannot be undone.')) {
        document.getElementById('process_action').value = 'end_process';
        document.getElementById('final_status').value = 'process_ended';
        submitInterviewProcess();
    }
}

function submitInterviewProcess() {
    // Get the form
    const form = document.getElementById('interviewProcessForm');
    
    // Add loading state to the clicked button
    if (event && event.target) {
        const submitBtn = event.target;
        submitBtn.classList.add('loading');
        submitBtn.disabled = true;
    }
    
    // Submit the form
    form.submit();
}

// Initialize the form when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Set minimum dates
    const today = new Date().toISOString().split('T')[0];
    const tomorrow = new Date(Date.now() + 24*60*60*1000).toISOString().split('T')[0];
    
    const nextLineupDate = document.getElementById('next_lineup_date');
    const nextInterviewDate = document.getElementById('next_interview_date');
    const joiningDate = document.getElementById('joining_date');
    const actualJoiningDate = document.getElementById('actual_joining_date');
    
    if (nextLineupDate) nextLineupDate.min = tomorrow;
    if (nextInterviewDate) nextInterviewDate.min = tomorrow;
    if (joiningDate) joiningDate.min = today;
    if (actualJoiningDate) actualJoiningDate.max = today;
});
</script>
