<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

include_once '../config/database.php';
include_once '../includes/session.php';
include_once '../classes/Candidate.php';
include_once '../classes/Interview.php';

// Check if user is logged in
if (!isLoggedIn()) {
    echo json_encode(['success' => false, 'message' => 'Not authenticated']);
    exit();
}

$database = new Database();
$db = $database->getConnection();

if (!$db) {
    echo json_encode(['success' => false, 'message' => 'Database connection failed']);
    exit();
}

$candidate_id = $_GET['candidate_id'] ?? '';

if (empty($candidate_id)) {
    echo json_encode(['success' => false, 'message' => 'Candidate ID is required']);
    exit();
}

try {
    // Get candidate details
    $candidate = new Candidate($db);
    $candidate->candidate_id = $candidate_id;
    
    if (!$candidate->readOne()) {
        echo json_encode(['success' => false, 'message' => 'Candidate not found']);
        exit();
    }

    // Get interview history with enhanced details
    $interview_query = "SELECT i.*, 
                               CASE 
                                   WHEN i.interviewer_name IS NOT NULL THEN i.interviewer_name
                                   WHEN u.username IS NOT NULL THEN u.username
                                   ELSE 'System'
                               END as interviewer_display_name
                        FROM interviews i 
                        LEFT JOIN users u ON i.interviewer_id = u.id
                        WHERE i.candidate_id = ? 
                        ORDER BY i.interview_date DESC, i.created_at DESC";
    $interview_stmt = $db->prepare($interview_query);
    $interview_stmt->execute([$candidate->id]);
    $interviews = $interview_stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get next scheduled interview if any
    $next_interview_query = "SELECT * FROM interviews 
                            WHERE candidate_id = ? 
                            AND status = 'scheduled' 
                            AND interview_date >= CURDATE()
                            ORDER BY interview_date ASC, interview_time ASC 
                            LIMIT 1";
    $next_stmt = $db->prepare($next_interview_query);
    $next_stmt->execute([$candidate->id]);
    $next_interview = $next_stmt->fetch(PDO::FETCH_ASSOC);

    $response = [
        'success' => true,
        'candidate' => [
            'id' => $candidate->id,
            'candidate_id' => $candidate->candidate_id,
            'name' => $candidate->name,
            'email' => $candidate->email,
            'phone' => $candidate->phone,
            'referral_code' => $candidate->referral_code ?? '',
            'lineup_date' => $candidate->lineup_date,
            'status' => $candidate->status,
            'added_date' => $candidate->added_date
        ],
        'interviews' => $interviews,
        'next_interview' => $next_interview
    ];

    echo json_encode($response);

} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
}

?>

<div class="interview-process-container">
    <!-- Candidate Information Header -->
    <div class="candidate-header mb-4">
        <div class="row">
            <div class="col-md-8">
                <h5 class="mb-1">
                    <i class="fas fa-user me-2"></i>
                    <?php echo htmlspecialchars($candidate->name); ?>
                    <span class="badge bg-primary ms-2"><?php echo $candidate->candidate_id; ?></span>
                </h5>
                <div class="candidate-details">
                    <p class="mb-1">
                        <i class="fas fa-phone text-muted me-1"></i><?php echo $candidate->phone; ?>
                        <span class="mx-2">|</span>
                        <i class="fas fa-envelope text-muted me-1"></i><?php echo $candidate->email; ?>
                    </p>
                    <?php if (!empty($candidate->referral_code)): ?>
                    <p class="mb-1">
                        <i class="fas fa-tag text-muted me-1"></i>
                        Referral Code: <span class="badge bg-info"><?php echo htmlspecialchars($candidate->referral_code); ?></span>
                    </p>
                    <?php endif; ?>
                    <p class="mb-1">
                        <i class="fas fa-calendar text-muted me-1"></i>
                        Lineup Date: <?php echo date('d-M-Y', strtotime($candidate->lineup_date)); ?>
                    </p>
                </div>
            </div>
            <div class="col-md-4 text-end">
                <div class="current-status">
                    <small class="text-muted">Current Status</small><br>
                    <span class="status-badge status-<?php echo $candidate->status; ?>">
                        <?php echo strtoupper(str_replace('_', ' ', $candidate->status)); ?>
                    </span>
                </div>
            </div>
        </div>
    </div>

    <!-- Interview History -->
    <?php if (!empty($interviews)): ?>
    <div class="interview-history mb-4">
        <h6><i class="fas fa-history me-2"></i>Interview History (<?php echo count($interviews); ?> Records)</h6>
        <div class="table-responsive">
            <table class="table table-sm table-striped">
                <thead class="table-dark">
                    <tr>
                        <th>Date</th>
                        <th>Time</th>
                        <th>Type</th>
                        <th>Duration</th>
                        <th>Status</th>
                        <th>Result</th>
                        <th>Interviewer</th>
                        <th>Notes</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($interviews as $int): ?>
                    <tr class="<?php echo $int['status'] === 'scheduled' ? 'table-info' : ''; ?>">
                        <td><?php echo date('d-M-Y', strtotime($int['interview_date'])); ?></td>
                        <td><?php echo $int['interview_time'] ? date('H:i', strtotime($int['interview_time'])) : 'N/A'; ?></td>
                        <td>
                            <span class="badge bg-secondary">
                                <?php echo ucfirst(str_replace('_', ' ', $int['interview_type'])); ?>
                            </span>
                        </td>
                        <td><?php echo $int['duration_minutes'] ? $int['duration_minutes'] . ' min' : 'N/A'; ?></td>
                        <td>
                            <span class="badge bg-<?php 
                                echo $int['status'] === 'completed' ? 'success' : 
                                    ($int['status'] === 'scheduled' ? 'primary' : 'warning'); 
                            ?>">
                                <?php echo ucfirst($int['status']); ?>
                            </span>
                        </td>
                        <td>
                            <span class="badge bg-<?php 
                                echo match($int['result']) {
                                    'selected' => 'success',
                                    'rejected', 'process_ended' => 'danger',
                                    'hold' => 'warning',
                                    'shortlisted' => 'info',
                                    'rescheduled' => 'secondary',
                                    'pending' => 'light text-dark',
                                    default => 'light text-dark'
                                };
                            ?>">
                                <?php echo ucfirst(str_replace('_', ' ', $int['result'])); ?>
                            </span>
                        </td>
                        <td><?php echo htmlspecialchars($int['interviewer_display_name'] ?? 'N/A'); ?></td>
                        <td>
                            <small><?php echo htmlspecialchars(substr($int['notes'] ?? '', 0, 50)) . (strlen($int['notes'] ?? '') > 50 ? '...' : ''); ?></small>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    <hr>
    <?php endif; ?>

    <!-- Complete Interview Process History -->
    <?php if (!empty($interviews)): ?>
    <div class="process-history mb-4">
        <h6><i class="fas fa-route me-2"></i>Complete Interview Process Journey</h6>
        <div class="timeline-container">
            <?php 
            // Get all process steps for this candidate
            $process_query = "SELECT 
                                i.*,
                                r.original_date, r.new_date, r.reason as reschedule_reason,
                                j.expected_joining_date, j.actual_joining_date, j.salary_offered, j.position,
                                csh.previous_status, csh.new_status, csh.reason as status_reason, csh.changed_at,
                                CASE 
                                    WHEN i.interviewer_name IS NOT NULL THEN i.interviewer_name
                                    WHEN u.username IS NOT NULL THEN u.username
                                    ELSE 'System'
                                END as interviewer_display_name
                              FROM interviews i 
                              LEFT JOIN users u ON i.interviewer_id = u.id
                              LEFT JOIN reschedules r ON i.candidate_id = r.candidate_id AND DATE(i.created_at) = DATE(r.created_at)
                              LEFT JOIN joining_details j ON i.candidate_id = j.candidate_id
                              LEFT JOIN candidate_status_history csh ON i.candidate_id = csh.candidate_id AND DATE(i.created_at) = DATE(csh.changed_at)
                              WHERE i.candidate_id = ? 
                              ORDER BY i.created_at ASC";
            $process_stmt = $db->prepare($process_query);
            $process_stmt->execute([$candidate->id]);
            $process_steps = $process_stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $step_number = 1;
            foreach ($process_steps as $step): 
            ?>
            <div class="timeline-step">
                <div class="timeline-marker step-<?php echo $step['result']; ?>">
                    <span class="step-number"><?php echo $step_number; ?></span>
                </div>
                <div class="timeline-content">
                    <div class="step-header">
                        <h6 class="mb-1">
                            <?php echo ucfirst(str_replace('_', ' ', $step['interview_type'])); ?> Interview
                            <span class="badge bg-<?php 
                                echo match($step['result']) {
                                    'selected' => 'success',
                                    'rejected', 'process_ended' => 'danger',
                                    'hold' => 'warning',
                                    'shortlisted' => 'info',
                                    'rescheduled' => 'secondary',
                                    'pending' => 'light text-dark',
                                    default => 'light text-dark'
                                };
                            ?> ms-2">
                                <?php echo ucfirst(str_replace('_', ' ', $step['result'])); ?>
                            </span>
                        </h6>
                        <small class="text-muted">
                            <?php echo date('d-M-Y H:i', strtotime($step['created_at'])); ?>
                            by <?php echo htmlspecialchars($step['interviewer_display_name']); ?>
                        </small>
                    </div>
                    
                    <div class="step-details mt-2">
                        <div class="row">
                            <div class="col-md-6">
                                <p class="mb-1"><strong>Date:</strong> <?php echo date('d-M-Y', strtotime($step['interview_date'])); ?></p>
                                <?php if ($step['interview_time']): ?>
                                <p class="mb-1"><strong>Time:</strong> <?php echo date('H:i', strtotime($step['interview_time'])); ?></p>
                                <?php endif; ?>
                                <?php if ($step['duration_minutes']): ?>
                                <p class="mb-1"><strong>Duration:</strong> <?php echo $step['duration_minutes']; ?> minutes</p>
                                <?php endif; ?>
                            </div>
                            <div class="col-md-6">
                                <p class="mb-1"><strong>Status:</strong> 
                                    <span class="badge bg-<?php echo $step['status'] === 'completed' ? 'success' : 'primary'; ?>">
                                        <?php echo ucfirst($step['status']); ?>
                                    </span>
                                </p>
                                <p class="mb-1"><strong>Interviewer:</strong> <?php echo htmlspecialchars($step['interviewer_display_name']); ?></p>
                            </div>
                        </div>
                        
                        <?php if ($step['notes']): ?>
                        <div class="step-notes mt-2">
                            <strong>Notes:</strong>
                            <p class="text-muted mb-0"><?php echo nl2br(htmlspecialchars($step['notes'])); ?></p>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($step['result'] === 'rescheduled' && $step['reschedule_reason']): ?>
                        <div class="reschedule-info mt-2 p-2 bg-warning bg-opacity-10 rounded">
                            <strong>Reschedule Reason:</strong> <?php echo htmlspecialchars($step['reschedule_reason']); ?>
                            <?php if ($step['new_date']): ?>
                            <br><strong>New Date:</strong> <?php echo date('d-M-Y', strtotime($step['new_date'])); ?>
                            <?php endif; ?>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($step['result'] === 'selected' && ($step['expected_joining_date'] || $step['salary_offered'])): ?>
                        <div class="joining-info mt-2 p-2 bg-success bg-opacity-10 rounded">
                            <strong>Selection Details:</strong>
                            <?php if ($step['position']): ?>
                            <br><strong>Position:</strong> <?php echo htmlspecialchars($step['position']); ?>
                            <?php endif; ?>
                            <?php if ($step['salary_offered']): ?>
                            <br><strong>Salary Offered:</strong> ₹<?php echo number_format($step['salary_offered']); ?>
                            <?php endif; ?>
                            <?php if ($step['expected_joining_date']): ?>
                            <br><strong>Expected Joining:</strong> <?php echo date('d-M-Y', strtotime($step['expected_joining_date'])); ?>
                            <?php endif; ?>
                            <?php if ($step['actual_joining_date']): ?>
                            <br><strong>Actual Joining:</strong> <?php echo date('d-M-Y', strtotime($step['actual_joining_date'])); ?>
                            <?php endif; ?>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <?php 
            $step_number++;
            endforeach; 
            ?>
        </div>
    </div>
    <?php endif; ?>

    <!-- Next Scheduled Interview Alert -->
    <?php if ($next_interview): ?>
    <div class="alert alert-info mb-4">
        <h6><i class="fas fa-calendar-check me-2"></i>Next Scheduled Interview</h6>
        <p class="mb-1">
            <strong>Date:</strong> <?php echo date('d-M-Y', strtotime($next_interview['interview_date'])); ?>
            <?php if ($next_interview['interview_time']): ?>
                at <?php echo date('H:i', strtotime($next_interview['interview_time'])); ?>
            <?php endif; ?>
        </p>
        <p class="mb-1">
            <strong>Type:</strong> <?php echo ucfirst(str_replace('_', ' ', $next_interview['interview_type'])); ?>
        </p>
        <p class="mb-0">
            <strong>Interviewer:</strong> <?php echo htmlspecialchars($next_interview['interviewer_name'] ?? 'TBD'); ?>
        </p>
    </div>
    <?php endif; ?>

    <!-- Interview Process Form -->
    <form method="POST" action="process-interview.php" id="interviewProcessForm">
        <input type="hidden" name="candidate_id" value="<?php echo $candidate->id; ?>">
        <input type="hidden" name="candidate_code" value="<?php echo $candidate->candidate_id; ?>">
        <input type="hidden" name="process_action" id="process_action">
        <input type="hidden" name="final_status" id="final_status">
        <input type="hidden" name="process_interview" value="1">
        
        <!-- Step 1: Turn Up Status -->
        <div class="step-section active" id="step1">
            <div class="step-header">
                <span class="step-number">1</span>
                <h6><i class="fas fa-user-check me-2"></i>Turn Up Status</h6>
            </div>
            <p class="text-muted mb-3">Did the candidate turn up for the scheduled interview?</p>
            
            <div class="turn-up-options">
                <div class="row">
                    <div class="col-md-6 mb-2">
                        <button type="button" class="btn btn-outline-success btn-lg w-100 turn-up-btn" data-turned-up="true">
                            <i class="fas fa-check-circle me-2"></i>
                            <div>
                                <strong>Yes - Turned Up</strong>
                                <small class="d-block">Candidate attended the interview</small>
                            </div>
                        </button>
                    </div>
                    <div class="col-md-6 mb-2">
                        <button type="button" class="btn btn-outline-danger btn-lg w-100 turn-up-btn" data-turned-up="false">
                            <i class="fas fa-times-circle me-2"></i>
                            <div>
                                <strong>No - Didn't Turn Up</strong>
                                <small class="d-block">Candidate was absent</small>
                            </div>
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Step 2A: If No - Reschedule or End Process Options -->
        <div class="step-section" id="step2a" style="display: none;">
            <div class="step-header">
                <span class="step-number">2</span>
                <h6><i class="fas fa-calendar-alt me-2"></i>Next Action</h6>
            </div>
            <p class="text-muted mb-3">What would you like to do since the candidate didn't turn up?</p>
            
            <div class="row mb-4">
                <div class="col-md-6">
                    <div class="card h-100">
                        <div class="card-body text-center">
                            <i class="fas fa-calendar-plus fa-3x text-warning mb-3"></i>
                            <h6>Reschedule Interview</h6>
                            <p class="text-muted small">Give the candidate another chance</p>
                            <button type="button" class="btn btn-warning" onclick="showRescheduleForm()">
                                <i class="fas fa-calendar-plus me-1"></i>Reschedule
                            </button>
                        </div>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="card h-100">
                        <div class="card-body text-center">
                            <i class="fas fa-stop-circle fa-3x text-danger mb-3"></i>
                            <h6>End Process</h6>
                            <p class="text-muted small">Stop the interview process</p>
                            <button type="button" class="btn btn-danger" onclick="showEndProcessForm()">
                                <i class="fas fa-stop me-1"></i>End Process
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Step 2A1: Reschedule Form -->
        <div class="step-section" id="step2a1" style="display: none;">
            <div class="step-header">
                <span class="step-number">2</span>
                <h6><i class="fas fa-calendar-plus me-2"></i>Reschedule Interview</h6>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="next_lineup_date" class="form-label">
                        <i class="fas fa-calendar me-1"></i>Next Lineup Date <span class="text-danger">*</span>
                    </label>
                    <input type="date" class="form-control" name="next_lineup_date" id="next_lineup_date" 
                           min="<?php echo date('Y-m-d', strtotime('+1 day')); ?>" required>
                </div>
                <div class="col-md-6">
                    <label for="not_coming_reason" class="form-label">
                        <i class="fas fa-question-circle me-1"></i>Reason for Not Coming <span class="text-danger">*</span>
                    </label>
                    <select class="form-select" name="not_coming_reason" id="not_coming_reason" required>
                        <option value="">Select Reason</option>
                        <option value="Personal Emergency">Personal Emergency</option>
                        <option value="Health Issues">Health Issues</option>
                        <option value="Transport Issues">Transport Issues</option>
                        <option value="Other Commitment">Other Commitment</option>
                        <option value="Changed Mind">Changed Mind</option>
                        <option value="Family Issues">Family Issues</option>
                        <option value="Weather Issues">Weather Issues</option>
                        <option value="Other">Other</option>
                    </select>
                </div>
            </div>
            
            <div class="mb-4">
                <label for="reason_details" class="form-label">
                    <i class="fas fa-sticky-note me-1"></i>Additional Details (Optional)
                </label>
                <textarea class="form-control" name="reason_details" id="reason_details" rows="3" 
                          placeholder="Any additional information about why the candidate didn't turn up..."></textarea>
            </div>
            
            <div class="text-center">
                <button type="button" class="btn btn-secondary me-2" onclick="goBackToOptions()">
                    <i class="fas fa-arrow-left me-1"></i>Back
                </button>
                <button type="button" class="btn btn-warning btn-lg" onclick="rescheduleCandidate()">
                    <i class="fas fa-calendar-plus me-2"></i>Reschedule Interview
                </button>
            </div>
        </div>

        <!-- Step 2A2: End Process Form -->
        <div class="step-section" id="step2a2" style="display: none;">
            <div class="step-header">
                <span class="step-number">2</span>
                <h6><i class="fas fa-stop-circle me-2"></i>End Interview Process</h6>
            </div>
            
            <div class="alert alert-warning">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <strong>Warning:</strong> This will permanently end the interview process for this candidate.
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="end_reason" class="form-label">
                        <i class="fas fa-question-circle me-1"></i>Reason for Ending Process <span class="text-danger">*</span>
                    </label>
                    <select class="form-select" name="not_coming_reason" id="end_reason" required>
                        <option value="">Select Reason</option>
                        <option value="No Show Multiple Times">No Show Multiple Times</option>
                        <option value="Candidate Not Interested">Candidate Not Interested</option>
                        <option value="Poor Communication">Poor Communication</option>
                        <option value="Not Meeting Requirements">Not Meeting Requirements</option>
                        <option value="Other">Other</option>
                    </select>
                </div>
            </div>
            
            <div class="mb-4">
                <label for="end_details" class="form-label">
                    <i class="fas fa-sticky-note me-1"></i>Additional Details (Optional)
                </label>
                <textarea class="form-control" name="reason_details" id="end_details" rows="3" 
                          placeholder="Any additional information about why the process is being ended..."></textarea>
            </div>
            
            <div class="text-center">
                <button type="button" class="btn btn-secondary me-2" onclick="goBackToOptions()">
                    <i class="fas fa-arrow-left me-1"></i>Back
                </button>
                <button type="button" class="btn btn-danger btn-lg" onclick="endProcess()">
                    <i class="fas fa-stop me-2"></i>End Process
                </button>
            </div>
        </div>

        <!-- Step 2B: If Yes - Interview Details -->
        <div class="step-section" id="step2b" style="display: none;">
            <div class="step-header">
                <span class="step-number">2</span>
                <h6><i class="fas fa-clipboard-list me-2"></i>Interview Details</h6>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="interview_type" class="form-label">
                        <i class="fas fa-list me-1"></i>Interview Type <span class="text-danger">*</span>
                    </label>
                    <select class="form-select" name="interview_type" id="interview_type" required>
                        <option value="">Select Interview Type</option>
                        <option value="initial">Initial Screening</option>
                        <option value="technical">Technical Interview</option>
                        <option value="hr">HR Interview</option>
                        <option value="final">Final Interview</option>
                        <option value="group_discussion">Group Discussion</option>
                        <option value="practical_test">Practical Test</option>
                        <option value="management">Management Round</option>
                    </select>
                </div>
                <div class="col-md-6">
                    <label for="interviewer_name" class="form-label">
                        <i class="fas fa-user-tie me-1"></i>Interviewer Name <span class="text-danger">*</span>
                    </label>
                    <input type="text" class="form-control" name="interviewer_name" id="interviewer_name" 
                           value="<?php echo htmlspecialchars($_SESSION['username'] ?? ''); ?>" required>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="interview_duration" class="form-label">
                        <i class="fas fa-clock me-1"></i>Interview Duration (minutes)
                    </label>
                    <input type="number" class="form-control" name="interview_duration" id="interview_duration" 
                           placeholder="e.g., 30" min="1" max="300">
                </div>
                <div class="col-md-6">
                    <label for="interview_mode" class="form-label">
                        <i class="fas fa-video me-1"></i>Interview Mode
                    </label>
                    <select class="form-select" name="interview_mode" id="interview_mode">
                        <option value="in_person">In Person</option>
                        <option value="video_call">Video Call</option>
                        <option value="phone_call">Phone Call</option>
                        <option value="online_test">Online Test</option>
                    </select>
                </div>
            </div>
            
            <div class="mb-4">
                <label for="interview_notes" class="form-label">
                    <i class="fas fa-edit me-1"></i>Interview Notes & Feedback
                </label>
                <textarea class="form-control" name="interview_notes" id="interview_notes" rows="4" 
                          placeholder="Interview feedback, observations, technical skills assessment, communication skills, overall impression..."></textarea>
            </div>
            
            <div class="text-center">
                <button type="button" class="btn btn-primary btn-lg" onclick="showInterviewResults()">
                    <i class="fas fa-arrow-right me-2"></i>Proceed to Interview Results
                </button>
            </div>
        </div>

        <!-- Step 3: Interview Results -->
        <div class="step-section" id="step3" style="display: none;">
            <div class="step-header">
                <span class="step-number">3</span>
                <h6><i class="fas fa-star me-2"></i>Interview Result</h6>
            </div>
            <p class="text-muted mb-4">Based on the interview, what is your decision for this candidate?</p>
            
            <div class="result-options">
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <button type="button" class="btn btn-outline-info btn-lg w-100 result-btn" data-result="shortlisted">
                            <i class="fas fa-list-ul me-2"></i>
                            <div>
                                <strong>Shortlisted</strong>
                                <small class="d-block">Move to next interview round</small>
                            </div>
                        </button>
                    </div>
                    <div class="col-md-6 mb-3">
                        <button type="button" class="btn btn-outline-success btn-lg w-100 result-btn" data-result="selected">
                            <i class="fas fa-check-circle me-2"></i>
                            <div>
                                <strong>Selected</strong>
                                <small class="d-block">Final selection for the role</small>
                            </div>
                        </button>
                    </div>
                    <div class="col-md-6 mb-3">
                        <button type="button" class="btn btn-outline-danger btn-lg w-100 result-btn" data-result="rejected">
                            <i class="fas fa-times-circle me-2"></i>
                            <div>
                                <strong>Rejected</strong>
                                <small class="d-block">Not suitable for the role</small>
                            </div>
                        </button>
                    </div>
                    <div class="col-md-6 mb-3">
                        <button type="button" class="btn btn-outline-warning btn-lg w-100 result-btn" data-result="hold">
                            <i class="fas fa-pause-circle me-2"></i>
                            <div>
                                <strong>Hold</strong>
                                <small class="d-block">Decision pending</small>
                            </div>
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Step 4A: If Shortlisted - Next Interview -->
        <div class="step-section" id="step4a" style="display: none;">
            <div class="step-header">
                <span class="step-number">4</span>
                <h6><i class="fas fa-calendar-check me-2"></i>Schedule Next Interview</h6>
            </div>
            
            <div class="alert alert-success mb-3">
                <i class="fas fa-check-circle me-2"></i>
                <strong>Candidate Shortlisted!</strong> Current interview will be recorded as completed with "shortlisted" result.
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="next_interview_date" class="form-label">
                        <i class="fas fa-calendar me-1"></i>Next Interview Date <span class="text-danger">*</span>
                    </label>
                    <input type="date" class="form-control" name="next_interview_date" id="next_interview_date" 
                           min="<?php echo date('Y-m-d', strtotime('+1 day')); ?>" required>
                </div>
                <div class="col-md-6">
                    <label for="next_interview_type" class="form-label">
                        <i class="fas fa-list me-1"></i>Next Interview Type <span class="text-danger">*</span>
                    </label>
                    <select class="form-select" name="next_interview_type" id="next_interview_type" required>
                        <option value="">Select Next Interview Type</option>
                        <option value="technical">Technical Interview</option>
                        <option value="hr">HR Interview</option>
                        <option value="final">Final Interview</option>
                        <option value="group_discussion">Group Discussion</option>
                        <option value="practical_test">Practical Test</option>
                        <option value="management">Management Round</option>
                    </select>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="next_interviewer" class="form-label">
                        <i class="fas fa-user-tie me-1"></i>Next Interviewer <span class="text-danger">*</span>
                    </label>
                    <input type="text" class="form-control" name="next_interviewer" id="next_interviewer" 
                           placeholder="Name of next interviewer" required>
                </div>
                <div class="col-md-6">
                    <label for="next_interview_time" class="form-label">
                        <i class="fas fa-clock me-1"></i>Interview Time
                    </label>
                    <input type="time" class="form-control" name="next_interview_time" id="next_interview_time">
                </div>
            </div>
            
            <div class="mb-4">
                <label for="next_interview_notes" class="form-label">
                    <i class="fas fa-sticky-note me-1"></i>Instructions for Next Interview
                </label>
                <textarea class="form-control" name="next_interview_notes" id="next_interview_notes" rows="3" 
                          placeholder="Special instructions, topics to focus on, documents to bring..."></textarea>
            </div>
            
            <div class="text-center">
                <button type="button" class="btn btn-success btn-lg" onclick="scheduleNextInterview()">
                    <i class="fas fa-calendar-plus me-2"></i>Schedule Next Interview
                </button>
            </div>
        </div>

        <!-- Step 4B: If Selected - Joining Process -->
        <div class="step-section" id="step4b" style="display: none;">
            <div class="step-header">
                <span class="step-number">4</span>
                <h6><i class="fas fa-handshake me-2"></i>Joining Process</h6>
            </div>
            
            <div class="alert alert-success mb-3">
                <i class="fas fa-check-circle me-2"></i>
                <strong>Candidate Selected!</strong> Interview will be recorded as completed with "selected" result.
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="joining_date" class="form-label">
                        <i class="fas fa-calendar me-1"></i>Expected Joining Date
                    </label>
                    <input type="date" class="form-control" name="joining_date" id="joining_date" 
                           min="<?php echo date('Y-m-d'); ?>">
                </div>
                <div class="col-md-6">
                    <label for="salary_offered" class="form-label">
                        <i class="fas fa-money-bill me-1"></i>Salary Offered (Annual)
                    </label>
                    <input type="number" class="form-control" name="salary_offered" id="salary_offered" 
                           placeholder="e.g., 500000">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="position" class="form-label">
                        <i class="fas fa-briefcase me-1"></i>Position/Designation
                    </label>
                    <input type="text" class="form-control" name="position" id="position" 
                           placeholder="e.g., Software Developer">
                </div>
                <div class="col-md-6">
                    <label for="department" class="form-label">
                        <i class="fas fa-building me-1"></i>Department
                    </label>
                    <input type="text" class="form-control" name="department" id="department" 
                           placeholder="e.g., IT Department">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="reporting_manager" class="form-label">
                        <i class="fas fa-user-tie me-1"></i>Reporting Manager
                    </label>
                    <input type="text" class="form-control" name="reporting_manager" id="reporting_manager" 
                           placeholder="Manager's name">
                </div>
                <div class="col-md-6">
                    <label for="work_location" class="form-label">
                        <i class="fas fa-map-marker-alt me-1"></i>Work Location
                    </label>
                    <input type="text" class="form-control" name="work_location" id="work_location" 
                           placeholder="Office location">
                </div>
            </div>
            
            <div class="mb-4">
                <label for="joining_notes" class="form-label">
                    <i class="fas fa-clipboard me-1"></i>Joining Instructions
                </label>
                <textarea class="form-control" name="joining_notes" id="joining_notes" rows="3" 
                          placeholder="Documents required, reporting details, first day instructions..."></textarea>
            </div>
            
            <div class="text-center">
                <button type="button" class="btn btn-success me-2" onclick="completeSelection()">
                    <i class="fas fa-check me-2"></i>Complete Selection Now
                </button>
                <button type="button" class="btn btn-primary btn-lg" onclick="setJoiningDetails()">
                    <i class="fas fa-calendar-check me-2"></i>Set Joining Details & Continue
                </button>
            </div>
        </div>

        <!-- Step 5: Joining Status -->
        <div class="step-section" id="step5" style="display: none;">
            <div class="step-header">
                <span class="step-number">5</span>
                <h6><i class="fas fa-user-plus me-2"></i>Joining Status</h6>
            </div>
            <p class="text-muted mb-3">Did the candidate join on the scheduled date?</p>
            
            <div class="mb-4">
                <label for="actual_joining_date" class="form-label">
                    <i class="fas fa-calendar me-1"></i>Actual Joining Date
                </label>
                <input type="date" class="form-control" name="actual_joining_date" id="actual_joining_date">
            </div>
            
            <div class="mb-4">
                <label for="joining_feedback" class="form-label">
                    <i class="fas fa-comment me-1"></i>Joining Feedback (Optional)
                </label>
                <textarea class="form-control" name="joining_feedback" id="joining_feedback" rows="2" 
                          placeholder="Any feedback about the joining process..."></textarea>
            </div>
            
            <div class="joining-options">
                <div class="row">
                    <div class="col-md-6 mb-2">
                        <button type="button" class="btn btn-outline-success btn-lg w-100 joining-btn" data-joined="true">
                            <i class="fas fa-check-circle me-2"></i>
                            <div>
                                <strong>Yes - Joined Successfully</strong>
                                <small class="d-block">Candidate joined the organization</small>
                            </div>
                        </button>
                    </div>
                    <div class="col-md-6 mb-2">
                        <button type="button" class="btn btn-outline-danger btn-lg w-100 joining-btn" data-joined="false">
                            <i class="fas fa-times-circle me-2"></i>
                            <div>
                                <strong>No - Didn't Join</strong>
                                <small class="d-block">Candidate didn't join</small>
                            </div>
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </form>

    <!-- Action Buttons -->
    <div class="text-center mt-4 pt-3 border-top">
        <button type="button" class="btn btn-secondary btn-lg" data-bs-dismiss="modal">
            <i class="fas fa-times me-2"></i>Cancel
        </button>
    </div>
</div>

<style>
.interview-process-container {
    max-height: 80vh;
    overflow-y: auto;
}

.candidate-header {
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    border-radius: 10px;
    padding: 20px;
    border-left: 4px solid #007bff;
}

.candidate-details p {
    margin-bottom: 0.5rem;
    color: #6c757d;
}

.current-status {
    text-align: center;
}

.step-section {
    background: #ffffff;
    border: 2px solid #e9ecef;
    border-radius: 15px;
    padding: 25px;
    margin-bottom: 20px;
    transition: all 0.3s ease;
    position: relative;
}

.step-section.active {
    border-color: #007bff;
    box-shadow: 0 0 20px rgba(0, 123, 255, 0.1);
}

.step-header {
    display: flex;
    align-items: center;
    margin-bottom: 15px;
}

.step-number {
    background: #007bff;
    color: white;
    width: 30px;
    height: 30px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    margin-right: 15px;
    font-size: 14px;
}

.step-header h6 {
    color: #007bff;
    margin: 0;
    font-weight: 600;
}

.turn-up-options .btn,
.result-options .btn,
.joining-options .btn {
    padding: 20px;
    text-align: left;
    border: 2px solid;
    transition: all 0.3s ease;
}

.turn-up-options .btn:hover,
.result-options .btn:hover,
.joining-options .btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(0,0,0,0.1);
}

.turn-up-options .btn.selected,
.result-options .btn.selected,
.joining-options .btn.selected {
    transform: scale(1.05);
    box-shadow: 0 5px 20px rgba(0,0,0,0.2);
}

.action-buttons {
    text-align: center;
}

.action-buttons .btn {
    margin: 0 10px;
    padding: 12px 30px;
}

.interview-history {
    background: #f8f9fa;
    border-radius: 10px;
    padding: 20px;
    border-left: 4px solid #28a745;
}

.status-badge {
    font-size: 0.75rem;
    padding: 6px 12px;
    border-radius: 20px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-pending { background-color: #fff3cd; color: #856404; }
.status-shortlisted { background-color: #d4edda; color: #155724; }
.status-selected { background-color: #d1ecf1; color: #0c5460; }
.status-rejected { background-color: #f8d7da; color: #721c24; }
.status-hold { background-color: #e2e3e5; color: #383d41; }
.status-joined { background-color: #d1ecf1; color: #0c5460; }
.status-rescheduled { background-color: #ffeaa7; color: #6c5ce7; }
.status-process-ended { background-color: #fab1a0; color: #e17055; }

.form-label {
    font-weight: 600;
    color: #495057;
    margin-bottom: 8px;
}

.form-control, .form-select {
    border-radius: 8px;
    border: 1px solid #ced4da;
    padding: 10px 15px;
}

.form-control:focus, .form-select:focus {
    border-color: #007bff;
    box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25);
}

.table-info {
    background-color: rgba(13, 202, 240, 0.1);
}

.card {
    transition: all 0.3s ease;
}

.card:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(0,0,0,0.1);
}

@media (max-width: 768px) {
    .step-section {
        padding: 15px;
    }
    
    .turn-up-options .btn,
    .result-options .btn,
    .joining-options .btn {
        padding: 15px;
        margin-bottom: 10px;
    }
    
    .action-buttons .btn {
        margin: 5px;
        padding: 10px 20px;
    }
}

/* Loading animation */
.btn.loading {
    position: relative;
    color: transparent;
}

.btn.loading::after {
    content: "";
    position: absolute;
    width: 16px;
    height: 16px;
    top: 50%;
    left: 50%;
    margin-left: -8px;
    margin-top: -8px;
    border: 2px solid transparent;
    border-top-color: #ffffff;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.process-history {
    background: #f8f9fa;
    border-radius: 10px;
    padding: 20px;
    border-left: 4px solid #6f42c1;
}

.timeline-container {
    position: relative;
    padding-left: 30px;
}

.timeline-container::before {
    content: '';
    position: absolute;
    left: 15px;
    top: 0;
    bottom: 0;
    width: 2px;
    background: linear-gradient(to bottom, #007bff, #6f42c1);
}

.timeline-step {
    position: relative;
    margin-bottom: 30px;
}

.timeline-marker {
    position: absolute;
    left: -23px;
    top: 5px;
    width: 30px;
    height: 30px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 12px;
    color: white;
    border: 3px solid white;
    box-shadow: 0 2px 5px rgba(0,0,0,0.2);
}

.timeline-marker.step-selected { background: #28a745; }
.timeline-marker.step-rejected { background: #dc3545; }
.timeline-marker.step-shortlisted { background: #17a2b8; }
.timeline-marker.step-hold { background: #ffc107; color: #000; }
.timeline-marker.step-rescheduled { background: #6c757d; }
.timeline-marker.step-process_ended { background: #e74c3c; }
.timeline-marker.step-pending { background: #007bff; }

.timeline-content {
    background: white;
    border-radius: 8px;
    padding: 15px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    border-left: 3px solid #007bff;
}

.step-header h6 {
    color: #495057;
    margin-bottom: 5px;
}

.step-details {
    font-size: 0.9rem;
}

.step-notes {
    background: #f8f9fa;
    padding: 10px;
    border-radius: 5px;
    border-left: 3px solid #007bff;
}

.reschedule-info, .joining-info {
    font-size: 0.85rem;
}

@media (max-width: 768px) {
    .timeline-container {
        padding-left: 20px;
    }
    
    .timeline-marker {
        left: -18px;
        width: 25px;
        height: 25px;
        font-size: 10px;
    }
}
</style>
