<?php
session_start();

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Not authenticated']);
    exit();
}

include_once '../config/database.php';

try {
    $database = new Database();
    $db = $database->getConnection();
    
    if (!$db) {
        throw new Exception('Database connection failed');
    }
    
    $candidate_id = $_POST['candidate_id'] ?? '';
    $current_step = $_POST['current_step'] ?? '';
    $turn_up_status = $_POST['turn_up_status'] ?? null;
    
    if (empty($candidate_id) || empty($current_step)) {
        throw new Exception('Missing required fields');
    }
    
    // Prepare data to save
    $step_data = [];
    
    // Collect step-specific data
    switch ($current_step) {
        case '1':
            $step_data['turn_up_status'] = $turn_up_status;
            break;
            
        case '2a1': // Reschedule form
            $step_data['next_lineup_date'] = $_POST['next_lineup_date'] ?? '';
            $step_data['not_coming_reason'] = $_POST['not_coming_reason'] ?? '';
            $step_data['reason_details'] = $_POST['reason_details'] ?? '';
            break;
            
        case '2a2': // End process form
            $step_data['end_reason'] = $_POST['not_coming_reason'] ?? '';
            $step_data['end_details'] = $_POST['reason_details'] ?? '';
            break;
            
        case '2b': // Interview result
            $step_data['interview_type'] = $_POST['interview_type'] ?? '';
            $step_data['interviewer_name'] = $_POST['interviewer_name'] ?? '';
            $step_data['interview_notes'] = $_POST['interview_notes'] ?? '';
            $step_data['interview_result'] = $_POST['interview_result'] ?? '';
            break;
            
        case '3a': // Rejection details
            $step_data['rejection_reason'] = $_POST['rejection_reason'] ?? '';
            $step_data['rejection_details'] = $_POST['rejection_details'] ?? '';
            break;
            
        case '3b': // Hold details
            $step_data['hold_reason'] = $_POST['hold_reason'] ?? '';
            $step_data['hold_details'] = $_POST['hold_details'] ?? '';
            break;
            
        case '3c': // Next interview round
            $step_data['next_interview_date'] = $_POST['next_interview_date'] ?? '';
            $step_data['next_interview_type'] = $_POST['next_interview_type'] ?? '';
            $step_data['next_interview_notes'] = $_POST['next_interview_notes'] ?? '';
            break;
            
        case '3d': // Offer rollout
            $step_data['offer_decision'] = $_POST['offer_decision'] ?? '';
            break;
            
        case '4': // Offer acceptance
            $step_data['offer_accepted'] = $_POST['offer_accepted'] ?? '';
            break;
            
        case '5': // Joining details
            $step_data['joining_date'] = $_POST['joining_date'] ?? '';
            $step_data['salary_offered'] = $_POST['salary_offered'] ?? '';
            $step_data['position'] = $_POST['position'] ?? '';
            $step_data['department'] = $_POST['department'] ?? '';
            $step_data['joining_notes'] = $_POST['joining_notes'] ?? '';
            break;
            
        case '6': // Joining confirmation
            $step_data['actual_joining_date'] = $_POST['actual_joining_date'] ?? '';
            $step_data['joining_feedback'] = $_POST['joining_feedback'] ?? '';
            $step_data['joined_status'] = $_POST['joined_status'] ?? '';
            break;
    }
    
    // Check if interview_process_state table exists, if not create it
    $create_table_query = "
        CREATE TABLE IF NOT EXISTS interview_process_state (
            id INT AUTO_INCREMENT PRIMARY KEY,
            candidate_id INT NOT NULL,
            current_step VARCHAR(10) NOT NULL,
            step_data JSON,
            notes TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            created_by INT,
            FOREIGN KEY (candidate_id) REFERENCES candidates(id) ON DELETE CASCADE,
            FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL
        )
    ";
    
    $db->exec($create_table_query);
    
    // Check if there's an existing state for this candidate
    $check_query = "SELECT id FROM interview_process_state WHERE candidate_id = ? ORDER BY created_at DESC LIMIT 1";
    $check_stmt = $db->prepare($check_query);
    $check_stmt->execute([$candidate_id]);
    $existing_state = $check_stmt->fetch(PDO::FETCH_ASSOC);
    
    $notes = "Step {$current_step} progress saved at " . date('Y-m-d H:i:s');
    
    if ($existing_state) {
        // Update existing state
        $update_query = "
            UPDATE interview_process_state 
            SET current_step = ?, 
                step_data = ?, 
                notes = ?,
                updated_at = CURRENT_TIMESTAMP
            WHERE id = ?
        ";
        
        $update_stmt = $db->prepare($update_query);
        $result = $update_stmt->execute([
            $current_step,
            json_encode($step_data),
            $notes,
            $existing_state['id']
        ]);
    } else {
        // Insert new state
        $insert_query = "
            INSERT INTO interview_process_state 
            (candidate_id, current_step, step_data, notes, created_by) 
            VALUES (?, ?, ?, ?, ?)
        ";
        
        $insert_stmt = $db->prepare($insert_query);
        $result = $insert_stmt->execute([
            $candidate_id,
            $current_step,
            json_encode($step_data),
            $notes,
            $_SESSION['user_id']
        ]);
    }
    
    if ($result) {
        echo json_encode([
            'success' => true, 
            'message' => 'Progress saved successfully',
            'step' => $current_step,
            'timestamp' => date('Y-m-d H:i:s')
        ]);
    } else {
        throw new Exception('Failed to save progress');
    }
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false, 
        'message' => $e->getMessage()
    ]);
}
?>
