<?php
include_once '../config/database.php';
include_once '../classes/Candidate.php';
include_once '../includes/session.php';

if (!isLoggedIn()) {
    header("Location: ../login.php");
    exit();
}

if (!isset($_GET['candidate_id'])) {
    echo '<div class="alert alert-danger">Candidate ID is required</div>';
    exit();
}

try {
    $database = new Database();
    $db = $database->getConnection();
    
    $candidate_id = $_GET['candidate_id'];
    
    // Get candidate details with source and profile names
    $query = "SELECT c.*, s.name as source_name, p.name as profile_name
              FROM candidates c
              LEFT JOIN sources s ON c.source_id = s.id
              LEFT JOIN profiles p ON c.profile_id = p.id
              WHERE c.candidate_id = ?";
    
    $stmt = $db->prepare($query);
    $stmt->execute([$candidate_id]);
    $candidate = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$candidate) {
        echo '<div class="alert alert-danger">Candidate not found</div>';
        exit();
    }
    
    // Get interview history
    $interview_query = "SELECT * FROM interviews WHERE candidate_id = ? ORDER BY interview_date DESC";
    $interview_stmt = $db->prepare($interview_query);
    $interview_stmt->execute([$candidate['id']]);
    $interviews = $interview_stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    echo '<div class="alert alert-danger">Error: ' . $e->getMessage() . '</div>';
    exit();
}
?>

<div class="container-fluid p-0">
    <div class="row g-0">
        <div class="col-md-6">
            <div class="p-3">
                <h6><i class="fas fa-user me-2"></i>Personal Information</h6>
                <table class="table table-sm table-borderless">
                    <tr>
                        <td><strong>Candidate ID:</strong></td>
                        <td><?php echo htmlspecialchars($candidate['candidate_id']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Name:</strong></td>
                        <td><?php echo htmlspecialchars($candidate['name']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Email:</strong></td>
                        <td><a href="mailto:<?php echo htmlspecialchars($candidate['email']); ?>"><?php echo htmlspecialchars($candidate['email']); ?></a></td>
                    </tr>
                    <tr>
                        <td><strong>Phone:</strong></td>
                        <td><a href="tel:<?php echo htmlspecialchars($candidate['phone']); ?>"><?php echo htmlspecialchars($candidate['phone']); ?></a></td>
                    </tr>
                    <tr>
                        <td><strong>Location:</strong></td>
                        <td><a href="tel:<?php echo htmlspecialchars($candidate['location']); ?>"><?php echo htmlspecialchars($candidate['location']); ?></a></td>
                    </tr>
                    <tr>
                        <td><strong>Team Name:</strong></td>
                        <td><a href="tel:<?php echo htmlspecialchars($candidate['team_name']); ?>"><?php echo htmlspecialchars($candidate['team_name']); ?></a></td>
                    </tr>
                    <tr>
                        <td><strong>Referral Code:</strong></td>
                        <td>
                            <?php if (!empty($candidate['referral_code'])): ?>
                                <span class="badge bg-info"><?php echo htmlspecialchars($candidate['referral_code']); ?></span>
                            <?php else: ?>
                                <span class="text-muted">N/A</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                </table>
            </div>
        </div>
        <div class="col-md-6">
            <div class="p-3">
                <h6><i class="fas fa-briefcase me-2"></i>Job Information</h6>
                <table class="table table-sm table-borderless">
                    <tr>
                        <td><strong>Source:</strong></td>
                        <td><?php echo htmlspecialchars($candidate['source_name'] ?? 'N/A'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Profile:</strong></td>
                        <td><?php echo htmlspecialchars($candidate['profile_name'] ?? 'N/A'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Lineup Date:</strong></td>
                        <td><?php echo date('d-M-Y', strtotime($candidate['lineup_date'])); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Added Date:</strong></td>
                        <td><?php echo date('d-M-Y', strtotime($candidate['added_date'])); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Current Status:</strong></td>
                        <td>
                            <span class="badge bg-<?php 
                                echo match($candidate['status']) {
                                    'pending' => 'warning',
                                    'shortlisted' => 'info',
                                    'selected' => 'success',
                                    'rejected' => 'danger',
                                    'hold' => 'secondary',
                                    'joined' => 'success',
                                    'rescheduled' => 'primary',
                                    default => 'secondary'
                                };
                            ?>"><?php echo strtoupper($candidate['status']); ?></span>
                        </td>
                    </tr>
                </table>
            </div>
        </div>
    </div>
    
    <hr>
    
    <div class="p-3">
        <h6><i class="fas fa-history me-2"></i>Interview History</h6>
        <?php if (empty($interviews)): ?>
            <div class="alert alert-info">
                <i class="fas fa-info-circle me-2"></i>No interview history available for this candidate.
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-sm table-striped">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Type</th>
                            <th>Result</th>
                            <th>Interviewer</th>
                            <th>Duration</th>
                            <th>Notes</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($interviews as $interview): ?>
                            <tr>
                                <td><?php echo date('d-M-Y', strtotime($interview['interview_date'])); ?></td>
                                <td><?php echo ucfirst($interview['interview_type']); ?></td>
                                <td>
                                    <span class="badge bg-<?php 
                                        echo match($interview['result']) {
                                            'shortlist' => 'info',
                                            'select' => 'success',
                                            'reject' => 'danger',
                                            'hold' => 'secondary',
                                            default => 'secondary'
                                        };
                                    ?>"><?php echo strtoupper($interview['result']); ?></span>
                                </td>
                                <td><?php echo htmlspecialchars($interview['interviewer_display_name'] ?? 'N/A'); ?></td>
                                <td><?php echo $interview['interview_duration'] ? $interview['interview_duration'] . ' min' : 'N/A'; ?></td>
                                <td><?php echo htmlspecialchars(substr($interview['notes'] ?? '', 0, 50)) . (strlen($interview['notes'] ?? '') > 50 ? '...' : ''); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
    
    <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
            <i class="fas fa-times me-1"></i>Close
        </button>
        <button type="button" class="btn btn-primary" onclick="editCandidate('<?php echo $candidate['candidate_id']; ?>')">
            <i class="fas fa-edit me-1"></i>Edit Details
        </button>
        <button type="button" class="btn btn-info" onclick="printCandidateDetails('<?php echo $candidate['candidate_id']; ?>', <?php echo htmlspecialchars(json_encode($candidate)); ?>, <?php echo htmlspecialchars(json_encode($interviews)); ?>)">
            <i class="fas fa-print me-1"></i>Print Details
        </button>
    </div>
</div>

<script>
function printCandidateDetails(candidateId, candidateData, interviewData) {
    const printWindow = window.open('', '_blank');
    const candidate = candidateData;
    const interviews = interviewData;
    
    const printContent = `
        <!DOCTYPE html>
        <html>
        <head>
            <title>Candidate Details - ${candidate.name}</title>
            <style>
                body { 
                    font-family: Arial, sans-serif; 
                    margin: 20px; 
                    line-height: 1.6; 
                    color: #333;
                }
                .header { 
                    text-align: center; 
                    margin-bottom: 30px; 
                    border-bottom: 3px solid #007bff; 
                    padding-bottom: 15px; 
                }
                .header h1 { 
                    color: #007bff; 
                    margin: 0; 
                    font-size: 28px;
                }
                .header h2 { 
                    color: #6c757d; 
                    margin: 5px 0 0 0; 
                    font-size: 18px;
                    font-weight: normal;
                }
                .section { 
                    margin-bottom: 30px; 
                    page-break-inside: avoid;
                }
                .section h3 { 
                    background: linear-gradient(135deg, #007bff, #0056b3);
                    color: white;
                    padding: 12px 15px; 
                    margin: 0 0 15px 0; 
                    border-radius: 5px;
                    font-size: 16px;
                }
                .info-grid { 
                    display: grid; 
                    grid-template-columns: 1fr 1fr; 
                    gap: 20px; 
                    margin-bottom: 20px;
                }
                .info-item { 
                    margin-bottom: 12px; 
                    padding: 8px;
                    background-color: #f8f9fa;
                    border-radius: 4px;
                }
                .info-item strong { 
                    display: inline-block; 
                    width: 140px; 
                    color: #495057;
                    font-weight: 600;
                }
                .info-item .value {
                    color: #212529;
                }
                .status { 
                    padding: 6px 12px; 
                    border-radius: 20px; 
                    font-size: 11px; 
                    font-weight: bold; 
                    text-transform: uppercase;
                    display: inline-block;
                }
                .status-pending { background-color: #fff3cd; color: #856404; }
                .status-shortlisted { background-color: #d4edda; color: #155724; }
                .status-selected { background-color: #d1ecf1; color: #0c5460; }
                .status-rejected { background-color: #f8d7da; color: #721c24; }
                .status-hold { background-color: #e2e3e5; color: #383d41; }
                .status-joined { background-color: #d1ecf1; color: #0c5460; }
                .status-rescheduled { background-color: #cce5ff; color: #004085; }
                .interview-table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-top: 15px;
                }
                .interview-table th,
                .interview-table td {
                    border: 1px solid #dee2e6;
                    padding: 10px;
                    text-align: left;
                }
                .interview-table th {
                    background-color: #f8f9fa;
                    font-weight: 600;
                    color: #495057;
                }
                .interview-table tr:nth-child(even) {
                    background-color: #f8f9fa;
                }
                .print-date { 
                    text-align: right; 
                    font-size: 12px; 
                    color: #6c757d; 
                    margin-top: 40px; 
                    border-top: 1px solid #dee2e6;
                    padding-top: 15px;
                }
                .company-info {
                    text-align: center;
                    margin-bottom: 20px;
                    color: #6c757d;
                    font-size: 14px;
                }
                @media print {
                    body { margin: 0; }
                    .section { page-break-inside: avoid; }
                    .interview-table { page-break-inside: avoid; }
                }
            </style>
        </head>
        <body>
            <div class="header">
                <h1>Candidate Management System</h1>
                <h2>Detailed Candidate Report</h2>
            </div>
            
            <div class="company-info">
                <strong>AI Groyal Co.</strong> | HR Portal System<br>
                Generated by: ${localStorage.getItem('username') || 'System Admin'}
            </div>
            
            <div class="section">
                <h3><i class="fas fa-user"></i> Personal Information</h3>
                <div class="info-grid">
                    <div class="info-item">
                        <strong>Full Name:</strong> 
                        <span class="value">${candidate.name}</span>
                    </div>
                    <div class="info-item">
                        <strong>Candidate ID:</strong> 
                        <span class="value">${candidate.candidate_id}</span>
                    </div>
                    <div class="info-item">
                        <strong>Email Address:</strong> 
                        <span class="value">${candidate.email}</span>
                    </div>
                    <div class="info-item">
                        <strong>Phone Number:</strong> 
                        <span class="value">${candidate.phone}</span>
                    </div>
                   <div class="info-item">
                        <strong>Location:</strong> 
                        <span class="value">${candidate.location}</span>
                    </div>
                    <div class="info-item">
                        <strong>Team Name:</strong> 
                        <span class="value">${candidate.team_name}</span>
                    </div>
                    <div class="info-item">
                        <strong>Referral Code:</strong> 
                        <span class="value">${candidate.referral_code || 'Not Provided'}</span>
                    </div>
                    <div class="info-item">
                        <strong>Current Status:</strong> 
                        <span class="status status-${candidate.status.replace('_', '-')}">${candidate.status.toUpperCase().replace('_', ' ')}</span>
                    </div>
                </div>
            </div>
            
            <div class="section">
                <h3><i class="fas fa-briefcase"></i> Application Details</h3>
                <div class="info-grid">
                    <div class="info-item">
                        <strong>Lineup Date:</strong> 
                        <span class="value">${new Date(candidate.lineup_date).toLocaleDateString('en-GB')}</span>
                    </div>
                    <div class="info-item">
                        <strong>Application Source:</strong> 
                        <span class="value">${candidate.source_name || 'Not Specified'}</span>
                    </div>
                    <div class="info-item">
                        <strong>Applied Profile:</strong> 
                        <span class="value">${candidate.profile_name || 'Not Specified'}</span>
                    </div>
                    <div class="info-item">
                        <strong>Registration Date:</strong> 
                        <span class="value">${new Date(candidate.added_date).toLocaleDateString('en-GB')}</span>
                    </div>
                </div>
            </div>
            
            ${interviews.length > 0 ? `
                <div class="section">
                    <h3><i class="fas fa-history"></i> Complete Interview History</h3>
                    <table class="interview-table">
                        <thead>
                            <tr>
                                <th>Interview Date</th>
                                <th>Interview Type</th>
                                <th>Result</th>
                                <th>Interviewer</th>
                                <th>Duration</th>
                                <th>Notes/Feedback</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${interviews.map(interview => `
                                <tr>
                                    <td>${new Date(interview.interview_date).toLocaleDateString('en-GB')}</td>
                                    <td>${interview.interview_type.charAt(0).toUpperCase() + interview.interview_type.slice(1)}</td>
                                    <td><span class="status status-${interview.result}">${interview.result.toUpperCase()}</span></td>
                                    <td>${interview.interviewer_display_name || 'Not Specified'}</td>
                                    <td>${interview.interview_duration ? interview.interview_duration + ' minutes' : 'Not Recorded'}</td>
                                    <td>${interview.notes || 'No notes provided'}</td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                </div>
            ` : `
                <div class="section">
                    <h3><i class="fas fa-history"></i> Interview History</h3>
                    <div class="info-item" style="text-align: center; padding: 20px; background-color: #e9ecef;">
                        <strong>No interview records found for this candidate.</strong>
                    </div>
                </div>
            `}
            
            <div class="print-date">
                <strong>Report Generated:</strong> ${new Date().toLocaleString('en-GB', {
                    weekday: 'long',
                    year: 'numeric',
                    month: 'long',
                    day: 'numeric',
                    hour: '2-digit',
                    minute: '2-digit'
                })}<br>
                <strong>System:</strong> Candidate Management System v2.0
            </div>
        </body>
        </html>
    `;
    
    printWindow.document.write(printContent);
    printWindow.document.close();
    
    // Wait for content to load then print
    printWindow.onload = function() {
        setTimeout(() => {
            printWindow.print();
        }, 500);
    };
}
</script>
