<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Start output buffering to prevent header issues
ob_start();

include_once 'config/database.php';
include_once 'includes/session.php';
include_once 'classes/Candidate.php';

// Check if user is logged in
if (!isLoggedIn()) {
    $_SESSION['error'] = 'Please login to access this page';
    header('Location: login.html');
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $_SESSION['error'] = 'Invalid request method';
    header('Location: search-candidates.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

if (!$db) {
    $_SESSION['error'] = 'Database connection failed';
    header('Location: search-candidates.php');
    exit();
}

// Initialize redirect URL
$redirect_url = 'search-candidates.php';

try {
    $db->beginTransaction();
    
    $candidate_id = $_POST['candidate_id'] ?? '';
    $candidate_code = $_POST['candidate_code'] ?? '';
    $process_action = $_POST['process_action'] ?? '';
    $final_status = $_POST['final_status'] ?? '';
    
    if (empty($candidate_id)) {
        throw new Exception('Candidate ID is required');
    }
    
    // Get candidate details using the internal ID
    $candidate_query = "SELECT * FROM candidates WHERE id = ?";
    $stmt = $db->prepare($candidate_query);
    $stmt->execute([$candidate_id]);
    $candidate = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$candidate) {
        throw new Exception('Candidate not found');
    }
    
    $success_message = '';
    $current_date = date('Y-m-d');
    $current_time = date('H:i:s');
    $user_id = $_SESSION['user_id'];
    $username = $_SESSION['username'] ?? 'System';
    
    // Function to validate interview type
    function validateInterviewType($type) {
        $allowed_types = ['initial', 'technical', 'hr', 'final', 'group_discussion', 'practical_test', 'management'];
        return in_array($type, $allowed_types) ? $type : 'initial';
    }
    
    // Function to insert status history with detailed tracking
    function insertStatusHistory($db, $candidate_id, $previous_status, $new_status, $user_id, $reason, $additional_data = null) {
        // First, get user details
        $user_query = "SELECT username FROM users WHERE id = ?";
        $stmt = $db->prepare($user_query);
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        $username = $user['username'] ?? 'System';
        
        $history_query = "INSERT INTO candidate_status_history 
                         (candidate_id, previous_status, new_status, changed_by, changed_by_name, reason, additional_data, changed_at) 
                         VALUES (?, ?, ?, ?, ?, ?, ?, NOW())";
        $stmt = $db->prepare($history_query);
        $stmt->execute([
            $candidate_id, 
            $previous_status, 
            $new_status, 
            $user_id, 
            $username, 
            $reason, 
            $additional_data ? json_encode($additional_data) : null
        ]);
        
        error_log("Status History Added: Candidate ID: $candidate_id, $previous_status -> $new_status, Reason: $reason");
    }
    
    // Function to insert comprehensive interview record
    function insertInterviewRecord($db, $candidate_id, $interview_data) {
        $interview_query = "INSERT INTO interviews 
                           (candidate_id, interview_date, interview_time, interview_type, duration_minutes, 
                            status, result, notes, feedback, interviewer_name, interviewer_id, 
                            technical_score, communication_score, overall_score, strengths, weaknesses, 
                            recommendations, next_interview_date, next_interview_type, next_interview_notes,
                            turn_up_status, absence_reason, absence_details, created_at) 
                           VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";
        
        $stmt = $db->prepare($interview_query);
        $result = $stmt->execute([
            $candidate_id,
            $interview_data['date'] ?? date('Y-m-d'),
            $interview_data['time'] ?? date('H:i:s'),
            $interview_data['type'] ?? 'initial',
            $interview_data['duration'] ?? null,
            $interview_data['status'] ?? 'completed',
            $interview_data['result'] ?? 'pending',
            $interview_data['notes'] ?? '',
            $interview_data['feedback'] ?? '',
            $interview_data['interviewer_name'] ?? 'System',
            $interview_data['interviewer_id'] ?? null,
            $interview_data['technical_score'] ?? null,
            $interview_data['communication_score'] ?? null,
            $interview_data['overall_score'] ?? null,
            $interview_data['strengths'] ?? '',
            $interview_data['weaknesses'] ?? '',
            $interview_data['recommendations'] ?? '',
            $interview_data['next_interview_date'] ?? null,
            $interview_data['next_interview_type'] ?? null,
            $interview_data['next_interview_notes'] ?? '',
            $interview_data['turn_up_status'] ?? 'turned_up',
            $interview_data['absence_reason'] ?? null,
            $interview_data['absence_details'] ?? null
        ]);
        
        $interview_id = $db->lastInsertId();
        error_log("Interview Record Added: ID: $interview_id, Candidate ID: $candidate_id, Type: " . ($interview_data['type'] ?? 'initial') . ", Result: " . ($interview_data['result'] ?? 'pending'));
        
        return $interview_id;
    }
    
    // Function to insert timeline record
    function insertTimelineRecord($db, $candidate_id, $interview_id, $step_data) {
        $timeline_query = "INSERT INTO interview_process_timeline 
                          (candidate_id, interview_id, step_number, step_name, step_type, step_status, 
                           step_data, started_at, completed_at, performed_by, performed_by_name, notes) 
                          VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?, ?, ?)";
        
        $stmt = $db->prepare($timeline_query);
        $stmt->execute([
            $candidate_id,
            $interview_id,
            $step_data['step_number'] ?? 1,
            $step_data['step_name'] ?? 'Unknown Step',
            $step_data['step_type'] ?? 'interview',
            $step_data['step_status'] ?? 'completed',
            json_encode($step_data['data'] ?? []),
            $step_data['started_at'] ?? date('Y-m-d H:i:s'),
            $step_data['performed_by'] ?? null,
            $step_data['performed_by_name'] ?? 'System',
            $step_data['notes'] ?? ''
        ]);
    }
    
    // Function to get the current interview round number
    function getInterviewRoundNumber($candidate_id) {
        global $db;
        $round_query = "SELECT COUNT(*) as round_count FROM interviews WHERE candidate_id = ? AND result IN ('shortlisted', 'selected', 'rejected', 'hold')";
        $stmt = $db->prepare($round_query);
        $stmt->execute([$candidate_id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return ($result['round_count'] ?? 0) + 1;
    }

    switch ($process_action) {
        case 'reschedule':
            $next_lineup_date = $_POST['next_lineup_date'] ?? '';
            $not_coming_reason = $_POST['not_coming_reason'] ?? '';
            $reason_details = $_POST['reason_details'] ?? '';
            
            if (empty($next_lineup_date)) {
                throw new Exception('Next lineup date is required');
            }
            
            // Update candidate status and lineup date
            $update_query = "UPDATE candidates SET lineup_date = ?, status = 'rescheduled', updated_at = NOW() WHERE id = ?";
            $stmt = $db->prepare($update_query);
            $stmt->execute([$next_lineup_date, $candidate_id]);
            
            // Insert comprehensive reschedule record
            $reschedule_query = "INSERT INTO reschedules 
                                (candidate_id, original_date, new_date, reason, reason_details, 
                                 requested_by, rescheduled_by, rescheduled_by_name, reschedule_count, created_at) 
                                VALUES (?, ?, ?, ?, ?, 'hr', ?, ?, 
                                        (SELECT COALESCE(MAX(reschedule_count), 0) + 1 FROM reschedules r2 WHERE r2.candidate_id = ?), 
                                        NOW())";
            $stmt = $db->prepare($reschedule_query);
            $stmt->execute([
                $candidate_id, 
                $candidate['lineup_date'], 
                $next_lineup_date, 
                $not_coming_reason, 
                $reason_details, 
                $user_id,
                $username,
                $candidate_id
            ]);
            
            // Insert interview record for reschedule
            $interview_id = insertInterviewRecord($db, $candidate_id, [
                'type' => 'initial',
                'result' => 'rescheduled',
                'notes' => "Rescheduled due to: $not_coming_reason. Details: $reason_details",
                'interviewer_name' => $username,
                'interviewer_id' => $user_id,
                'turn_up_status' => 'no_show',
                'absence_reason' => $not_coming_reason,
                'absence_details' => $reason_details
            ]);
            
            // Insert timeline record
            insertTimelineRecord($db, $candidate_id, $interview_id, [
                'step_number' => 1,
                'step_name' => 'Interview Rescheduled',
                'step_type' => 'reschedule',
                'step_status' => 'completed',
                'data' => [
                    'original_date' => $candidate['lineup_date'],
                    'new_date' => $next_lineup_date,
                    'reason' => $not_coming_reason,
                    'details' => $reason_details
                ],
                'performed_by' => $user_id,
                'performed_by_name' => $username,
                'notes' => "Interview rescheduled from {$candidate['lineup_date']} to $next_lineup_date"
            ]);
            
            // Insert status history
            insertStatusHistory($db, $candidate_id, $candidate['status'], 'rescheduled', $user_id, 
                "Interview rescheduled to $next_lineup_date due to: $not_coming_reason", 
                [
                    'original_date' => $candidate['lineup_date'],
                    'new_date' => $next_lineup_date,
                    'reason' => $not_coming_reason,
                    'details' => $reason_details
                ]
            );
            
            $success_message = "Candidate rescheduled successfully for " . date('d-M-Y', strtotime($next_lineup_date));
            break;
            
        case 'end_process':
            $not_coming_reason = $_POST['not_coming_reason'] ?? '';
            $reason_details = $_POST['reason_details'] ?? '';
            
            // Update candidate status
            $update_query = "UPDATE candidates SET status = 'process_ended', updated_at = NOW() WHERE id = ?";
            $stmt = $db->prepare($update_query);
            $stmt->execute([$candidate_id]);
            
            // Insert interview record for process end
            $interview_id = insertInterviewRecord($db, $candidate_id, [
                'type' => 'initial',
                'result' => 'process_ended',
                'notes' => "Process ended - Reason: $not_coming_reason. Details: $reason_details",
                'interviewer_name' => $username,
                'interviewer_id' => $user_id,
                'turn_up_status' => 'no_show',
                'absence_reason' => $not_coming_reason,
                'absence_details' => $reason_details
            ]);
            
            // Insert timeline record
            insertTimelineRecord($db, $candidate_id, $interview_id, [
                'step_number' => 1,
                'step_name' => 'Process Ended',
                'step_type' => 'end_process',
                'step_status' => 'completed',
                'data' => [
                    'reason' => $not_coming_reason,
                    'details' => $reason_details
                ],
                'performed_by' => $user_id,
                'performed_by_name' => $username,
                'notes' => "Interview process ended due to: $not_coming_reason"
            ]);
            
            // Update process completion
            $process_query = "INSERT INTO process_completion 
                             (candidate_id, process_status, completion_reason, final_result, 
                              started_date, completed_date, completed_by, completed_by_name, process_notes) 
                             VALUES (?, 'terminated', ?, 'withdrawn', ?, NOW(), ?, ?, ?)
                             ON DUPLICATE KEY UPDATE 
                             process_status = 'terminated', completion_reason = ?, final_result = 'withdrawn',
                             completed_date = NOW(), completed_by = ?, completed_by_name = ?, process_notes = ?";
            $stmt = $db->prepare($process_query);
            $stmt->execute([
                $candidate_id, $not_coming_reason, $candidate['added_date'], $user_id, $username, $reason_details,
                $not_coming_reason, $user_id, $username, $reason_details
            ]);
            
            // Insert status history
            insertStatusHistory($db, $candidate_id, $candidate['status'], 'process_ended', $user_id, 
                "Interview process ended - Reason: $not_coming_reason", 
                [
                    'reason' => $not_coming_reason,
                    'details' => $reason_details
                ]
            );
            
            $success_message = "Interview process ended for candidate";
            break;
            
        case 'reject':
            // Get comprehensive interview details
            $interview_type = validateInterviewType($_POST['interview_type'] ?? 'initial');
            $interviewer_name = $_POST['interviewer_name'] ?? $username;
            $interview_notes = $_POST['interview_notes'] ?? '';
            $interview_duration = $_POST['interview_duration'] ?? null;
            $rejection_reason = $_POST['rejection_reason'] ?? '';
            $rejection_details = $_POST['rejection_details'] ?? '';
            $technical_score = $_POST['technical_score'] ?? null;
            $communication_score = $_POST['communication_score'] ?? null;
            $overall_score = $_POST['overall_score'] ?? null;
            $strengths = $_POST['strengths'] ?? '';
            $weaknesses = $_POST['weaknesses'] ?? '';
            
            // Update candidate status
            $update_query = "UPDATE candidates SET status = 'rejected', updated_at = NOW() WHERE id = ?";
            $stmt = $db->prepare($update_query);
            if (!$stmt->execute([$candidate_id])) {
                throw new Exception('Failed to update candidate status to rejected');
            }
            
            // Insert comprehensive interview record
            $full_notes = $interview_notes;
            if ($rejection_reason) {
                $full_notes .= "\n\nRejection Reason: $rejection_reason";
            }
            if ($rejection_details) {
                $full_notes .= "\nDetails: $rejection_details";
            }
            
            $interview_id = insertInterviewRecord($db, $candidate_id, [
                'type' => $interview_type,
                'duration' => $interview_duration,
                'result' => 'rejected',
                'notes' => $full_notes,
                'feedback' => $rejection_details,
                'interviewer_name' => $interviewer_name,
                'interviewer_id' => $user_id,
                'technical_score' => $technical_score,
                'communication_score' => $communication_score,
                'overall_score' => $overall_score,
                'strengths' => $strengths,
                'weaknesses' => $weaknesses,
                'recommendations' => "Not recommended for hire - $rejection_reason"
            ]);
            
            // Insert detailed feedback record
            if ($interview_id) {
                $feedback_query = "INSERT INTO interview_feedback 
                                  (interview_id, candidate_id, interviewer_id, interviewer_name, 
                                   technical_skills_score, communication_score, overall_score, 
                                   strengths, weaknesses, detailed_feedback, recommendations, 
                                   would_hire, feedback_date) 
                                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'no', ?)";
                $stmt = $db->prepare($feedback_query);
                $stmt->execute([
                    $interview_id, $candidate_id, $user_id, $interviewer_name,
                    $technical_score, $communication_score, $overall_score,
                    $strengths, $weaknesses, $full_notes, "Not recommended - $rejection_reason",
                    date('Y-m-d')
                ]);
            }
            
            // Insert timeline record
            insertTimelineRecord($db, $candidate_id, $interview_id, [
                'step_number' => getInterviewRoundNumber($candidate_id),
                'step_name' => ucfirst($interview_type) . ' Interview - Rejected',
                'step_type' => 'result',
                'step_status' => 'completed',
                'data' => [
                    'interview_type' => $interview_type,
                    'result' => 'rejected',
                    'rejection_reason' => $rejection_reason,
                    'scores' => [
                        'technical' => $technical_score,
                        'communication' => $communication_score,
                        'overall' => $overall_score
                    ]
                ],
                'performed_by' => $user_id,
                'performed_by_name' => $interviewer_name,
                'notes' => "Candidate rejected after $interview_type interview"
            ]);
            
            // Update process completion
            $process_query = "INSERT INTO process_completion 
                             (candidate_id, process_status, completion_reason, final_result, 
                              started_date, completed_date, completed_by, completed_by_name, process_notes) 
                             VALUES (?, 'completed', ?, 'rejected', ?, NOW(), ?, ?, ?)
                             ON DUPLICATE KEY UPDATE 
                             process_status = 'completed', completion_reason = ?, final_result = 'rejected',
                             completed_date = NOW(), completed_by = ?, completed_by_name = ?, process_notes = ?";
            $stmt = $db->prepare($process_query);
            $stmt->execute([
                $candidate_id, $rejection_reason, $candidate['added_date'], $user_id, $interviewer_name, $rejection_details,
                $rejection_reason, $user_id, $interviewer_name, $rejection_details
            ]);
            
            // Insert status history
            insertStatusHistory($db, $candidate_id, $candidate['status'], 'rejected', $user_id, 
                "Candidate rejected after $interview_type interview - Reason: $rejection_reason", 
                [
                    'interview_type' => $interview_type,
                    'rejection_reason' => $rejection_reason,
                    'rejection_details' => $rejection_details,
                    'interviewer' => $interviewer_name,
                    'scores' => [
                        'technical' => $technical_score,
                        'communication' => $communication_score,
                        'overall' => $overall_score
                    ]
                ]
            );
            
            $success_message = "Candidate rejected successfully";
            break;
            
        case 'hold':
            // Get comprehensive interview details
            $interview_type = validateInterviewType($_POST['interview_type'] ?? 'initial');
            $interviewer_name = $_POST['interviewer_name'] ?? $username;
            $interview_notes = $_POST['interview_notes'] ?? '';
            $interview_duration = $_POST['interview_duration'] ?? null;
            $hold_reason = $_POST['hold_reason'] ?? '';
            $hold_details = $_POST['hold_details'] ?? '';
            $technical_score = $_POST['technical_score'] ?? null;
            $communication_score = $_POST['communication_score'] ?? null;
            $overall_score = $_POST['overall_score'] ?? null;
            
            // Update candidate status
            $update_query = "UPDATE candidates SET status = 'hold', updated_at = NOW() WHERE id = ?";
            $stmt = $db->prepare($update_query);
            if (!$stmt->execute([$candidate_id])) {
                throw new Exception('Failed to update candidate status to hold');
            }
            
            // Insert comprehensive interview record
            $full_notes = $interview_notes;
            if ($hold_reason) {
                $full_notes .= "\n\nHold Reason: $hold_reason";
            }
            if ($hold_details) {
                $full_notes .= "\nDetails: $hold_details";
            }
            
            $interview_id = insertInterviewRecord($db, $candidate_id, [
                'type' => $interview_type,
                'duration' => $interview_duration,
                'result' => 'hold',
                'notes' => $full_notes,
                'feedback' => $hold_details,
                'interviewer_name' => $interviewer_name,
                'interviewer_id' => $user_id,
                'technical_score' => $technical_score,
                'communication_score' => $communication_score,
                'overall_score' => $overall_score,
                'recommendations' => "On hold - $hold_reason"
            ]);
            
            // Insert timeline record
            insertTimelineRecord($db, $candidate_id, $interview_id, [
                'step_number' => getInterviewRoundNumber($candidate_id),
                'step_name' => ucfirst($interview_type) . ' Interview - On Hold',
                'step_type' => 'result',
                'step_status' => 'completed',
                'data' => [
                    'interview_type' => $interview_type,
                    'result' => 'hold',
                    'hold_reason' => $hold_reason,
                    'scores' => [
                        'technical' => $technical_score,
                        'communication' => $communication_score,
                        'overall' => $overall_score
                    ]
                ],
                'performed_by' => $user_id,
                'performed_by_name' => $interviewer_name,
                'notes' => "Candidate put on hold after $interview_type interview"
            ]);
            
            // Insert status history
            insertStatusHistory($db, $candidate_id, $candidate['status'], 'hold', $user_id, 
                "Candidate put on hold after $interview_type interview - Reason: $hold_reason", 
                [
                    'interview_type' => $interview_type,
                    'hold_reason' => $hold_reason,
                    'hold_details' => $hold_details,
                    'interviewer' => $interviewer_name
                ]
            );
            
            $success_message = "Candidate put on hold successfully";
            break;
            
        case 'shortlist':
            // Get comprehensive current interview details
            $interview_type = validateInterviewType($_POST['interview_type'] ?? 'initial');
            $interviewer_name = $_POST['interviewer_name'] ?? $username;
            $interview_notes = $_POST['interview_notes'] ?? '';
            $interview_duration = $_POST['interview_duration'] ?? null;
            $technical_score = $_POST['technical_score'] ?? null;
            $communication_score = $_POST['communication_score'] ?? null;
            $overall_score = $_POST['overall_score'] ?? null;
            $strengths = $_POST['strengths'] ?? '';
            $weaknesses = $_POST['weaknesses'] ?? '';
            
            // Update candidate status to shortlisted
            $update_query = "UPDATE candidates SET status = 'shortlisted', updated_at = NOW() WHERE id = ?";
            $stmt = $db->prepare($update_query);
            $stmt->execute([$candidate_id]);
            
            // Insert current interview record
            $interview_id = insertInterviewRecord($db, $candidate_id, [
                'type' => $interview_type,
                'duration' => $interview_duration,
                'result' => 'shortlisted',
                'notes' => $interview_notes,
                'interviewer_name' => $interviewer_name,
                'interviewer_id' => $user_id,
                'technical_score' => $technical_score,
                'communication_score' => $communication_score,
                'overall_score' => $overall_score,
                'strengths' => $strengths,
                'weaknesses' => $weaknesses,
                'recommendations' => "Shortlisted for next round"
            ]);
            
            // Schedule next interview if provided
            $next_interview_date = $_POST['next_interview_date'] ?? '';
            $next_interview_type = $_POST['next_interview_type'] ?? '';
            $next_interviewer = $_POST['next_interviewer'] ?? '';
            $next_interview_notes = $_POST['next_interview_notes'] ?? '';
            
            if (!empty($next_interview_date) && !empty($next_interview_type)) {
                $next_interview_type = validateInterviewType($next_interview_type);
                
                // Update current interview with next round details
                $update_interview_query = "UPDATE interviews SET 
                                          next_interview_date = ?, 
                                          next_interview_type = ?, 
                                          next_interview_notes = ? 
                                          WHERE id = ?";
                $stmt = $db->prepare($update_interview_query);
                $stmt->execute([$next_interview_date, $next_interview_type, $next_interview_notes, $interview_id]);
                
                // Insert next interview record as scheduled
                insertInterviewRecord($db, $candidate_id, [
                    'date' => $next_interview_date,
                    'type' => $next_interview_type,
                    'status' => 'scheduled',
                    'result' => 'pending',
                    'notes' => "Next interview scheduled. Notes: $next_interview_notes",
                    'interviewer_name' => $next_interviewer ?: 'TBD'
                ]);
                
                // Update candidate lineup date for next round
                $update_lineup_query = "UPDATE candidates SET lineup_date = ? WHERE id = ?";
                $stmt = $db->prepare($update_lineup_query);
                $stmt->execute([$next_interview_date, $candidate_id]);
                
                $success_message = "Candidate shortlisted successfully and next $next_interview_type interview scheduled for " . date('d-M-Y', strtotime($next_interview_date));
            } else {
                $success_message = "Candidate shortlisted successfully - Next interview round to be scheduled";
            }
            
            // Insert timeline record
            insertTimelineRecord($db, $candidate_id, $interview_id, [
                'step_number' => getInterviewRoundNumber($candidate_id),
                'step_name' => ucfirst($interview_type) . ' Interview - Shortlisted',
                'step_type' => 'result',
                'step_status' => 'completed',
                'data' => [
                    'interview_type' => $interview_type,
                    'result' => 'shortlisted',
                    'next_interview_date' => $next_interview_date,
                    'next_interview_type' => $next_interview_type,
                    'scores' => [
                        'technical' => $technical_score,
                        'communication' => $communication_score,
                        'overall' => $overall_score
                    ]
                ],
                'performed_by' => $user_id,
                'performed_by_name' => $interviewer_name,
                'notes' => "Candidate shortlisted for next round"
            ]);
            
            // Insert status history
            insertStatusHistory($db, $candidate_id, $candidate['status'], 'shortlisted', $user_id, 
                "Candidate shortlisted after $interview_type interview" . 
                ($next_interview_date ? " - Next round: $next_interview_type on $next_interview_date" : ""), 
                [
                    'current_interview_type' => $interview_type,
                    'current_interviewer' => $interviewer_name,
                    'next_interview_date' => $next_interview_date,
                    'next_interview_type' => $next_interview_type,
                    'next_interviewer' => $next_interviewer,
                    'round_number' => getInterviewRoundNumber($candidate_id),
                    'scores' => [
                        'technical' => $technical_score,
                        'communication' => $communication_score,
                        'overall' => $overall_score
                    ]
                ]
            );
            
            break;
            
        case 'select':
        case 'select_hold_offer':
            // Get comprehensive interview details
            $interview_type = validateInterviewType($_POST['interview_type'] ?? 'final');
            $interviewer_name = $_POST['interviewer_name'] ?? $username;
            $interview_notes = $_POST['interview_notes'] ?? '';
            $interview_duration = $_POST['interview_duration'] ?? null;
            $technical_score = $_POST['technical_score'] ?? null;
            $communication_score = $_POST['communication_score'] ?? null;
            $overall_score = $_POST['overall_score'] ?? null;
            $strengths = $_POST['strengths'] ?? '';
            $weaknesses = $_POST['weaknesses'] ?? '';
            
            // Update candidate status
            $update_query = "UPDATE candidates SET status = 'selected', updated_at = NOW() WHERE id = ?";
            $stmt = $db->prepare($update_query);
            if (!$stmt->execute([$candidate_id])) {
                throw new Exception('Failed to update candidate status to selected');
            }
            
            // Insert comprehensive interview record
            $interview_id = insertInterviewRecord($db, $candidate_id, [
                'type' => $interview_type,
                'duration' => $interview_duration,
                'result' => 'selected',
                'notes' => $interview_notes,
                'interviewer_name' => $interviewer_name,
                'interviewer_id' => $user_id,
                'technical_score' => $technical_score,
                'communication_score' => $communication_score,
                'overall_score' => $overall_score,
                'strengths' => $strengths,
                'weaknesses' => $weaknesses,
                'recommendations' => "Recommended for hire"
            ]);
            
            // Insert detailed feedback record
            if ($interview_id) {
                $feedback_query = "INSERT INTO interview_feedback 
                                  (interview_id, candidate_id, interviewer_id, interviewer_name, 
                                   technical_skills_score, communication_score, overall_score, 
                                   strengths, weaknesses, detailed_feedback, recommendations, 
                                   would_hire, feedback_date) 
                                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'yes', ?)";
                $stmt = $db->prepare($feedback_query);
                $stmt->execute([
                    $interview_id, $candidate_id, $user_id, $interviewer_name,
                    $technical_score, $communication_score, $overall_score,
                    $strengths, $weaknesses, $interview_notes, "Recommended for hire",
                    date('Y-m-d')
                ]);
            }
            
            // Insert timeline record
            insertTimelineRecord($db, $candidate_id, $interview_id, [
                'step_number' => getInterviewRoundNumber($candidate_id),
                'step_name' => ucfirst($interview_type) . ' Interview - Selected',
                'step_type' => 'result',
                'step_status' => 'completed',
                'data' => [
                    'interview_type' => $interview_type,
                    'result' => 'selected',
                    'offer_status' => $process_action === 'select_hold_offer' ? 'on_hold' : 'to_be_rolled_out',
                    'scores' => [
                        'technical' => $technical_score,
                        'communication' => $communication_score,
                        'overall' => $overall_score
                    ]
                ],
                'performed_by' => $user_id,
                'performed_by_name' => $interviewer_name,
                'notes' => "Candidate selected for the position"
            ]);
            
            // Insert status history
            $reason = $process_action === 'select_hold_offer' ? 
                "Candidate selected after $interview_type interview - Offer on hold" : 
                "Candidate selected after $interview_type interview";
            insertStatusHistory($db, $candidate_id, $candidate['status'], 'selected', $user_id, $reason, 
                [
                    'interview_type' => $interview_type,
                    'interviewer' => $interviewer_name,
                    'offer_status' => $process_action === 'select_hold_offer' ? 'on_hold' : 'to_be_rolled_out',
                    'scores' => [
                        'technical' => $technical_score,
                        'communication' => $communication_score,
                        'overall' => $overall_score
                    ]
                ]
            );
            
            $success_message = $process_action === 'select_hold_offer' ? 
                "Candidate selected successfully - Offer on hold" : 
                "Candidate selected successfully";
            break;
            
        case 'offer_declined':
            // Update candidate status
            $update_query = "UPDATE candidates SET status = 'offer_declined', updated_at = NOW() WHERE id = ?";
            $stmt = $db->prepare($update_query);
            $stmt->execute([$candidate_id]);
            
            // Insert timeline record
            insertTimelineRecord($db, $candidate_id, null, [
                'step_number' => 5,
                'step_name' => 'Offer Declined',
                'step_type' => 'offer',
                'step_status' => 'completed',
                'data' => [
                    'offer_status' => 'declined',
                    'declined_date' => date('Y-m-d')
                ],
                'performed_by' => $user_id,
                'performed_by_name' => $username,
                'notes' => "Candidate declined the job offer"
            ]);
            
            // Update process completion
            $process_query = "INSERT INTO process_completion 
                             (candidate_id, process_status, completion_reason, final_result, 
                              started_date, completed_date, completed_by, completed_by_name, process_notes) 
                             VALUES (?, 'completed', 'offer_declined', 'offer_declined', ?, NOW(), ?, ?, ?)
                             ON DUPLICATE KEY UPDATE 
                             process_status = 'completed', completion_reason = 'offer_declined', final_result = 'offer_declined',
                             completed_date = NOW(), completed_by = ?, completed_by_name = ?, process_notes = ?";
            $stmt = $db->prepare($process_query);
            $stmt->execute([
                $candidate_id, $candidate['added_date'], $user_id, $username, 'Candidate declined job offer',
                $user_id, $username, 'Candidate declined job offer'
            ]);
            
            // Insert status history
            insertStatusHistory($db, $candidate_id, $candidate['status'], 'offer_declined', $user_id, 
                "Candidate declined the job offer", 
                [
                    'offer_declined_date' => date('Y-m-d')
                ]
            );
            
            $success_message = "Candidate offer declined status updated";
            break;
            
        case 'joined':
            // Get comprehensive joining details
            $joining_date = $_POST['joining_date'] ?? '';
            $actual_joining_date = $_POST['actual_joining_date'] ?? '';
            $salary_offered = $_POST['salary_offered'] ?? null;
            $salary_negotiated = $_POST['salary_negotiated'] ?? null;
            $position = $_POST['position'] ?? '';
            $department = $_POST['department'] ?? '';
            $reporting_manager = $_POST['reporting_manager'] ?? '';
            $work_location = $_POST['work_location'] ?? '';
            $employment_type = $_POST['employment_type'] ?? 'full_time';
            $joining_notes = $_POST['joining_notes'] ?? '';
            $joining_feedback = $_POST['joining_feedback'] ?? '';
            $documents_required = $_POST['documents_required'] ?? '';
            $probation_period = $_POST['probation_period_months'] ?? 6;
            
            // Update candidate status
            $update_query = "UPDATE candidates SET status = 'joined', updated_at = NOW() WHERE id = ?";
            $stmt = $db->prepare($update_query);
            $stmt->execute([$candidate_id]);
            
            // Insert or update comprehensive joining details
            $joining_check_query = "SELECT id FROM joining_details WHERE candidate_id = ?";
            $stmt = $db->prepare($joining_check_query);
            $stmt->execute([$candidate_id]);
            $existing_joining = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existing_joining) {
                // Update existing joining details
                $joining_update_query = "UPDATE joining_details SET 
                                        actual_joining_date = ?, 
                                        salary_negotiated = ?,
                                        reporting_manager = ?,
                                        work_location = ?,
                                        employment_type = ?,
                                        joining_status = 'joined', 
                                        joining_feedback = ?,
                                        documents_required = ?,
                                        probation_period_months = ?,
                                        updated_at = NOW() 
                                        WHERE candidate_id = ?";
                $stmt = $db->prepare($joining_update_query);
                $stmt->execute([
                    $actual_joining_date, $salary_negotiated, $reporting_manager, 
                    $work_location, $employment_type, $joining_feedback, 
                    $documents_required, $probation_period, $candidate_id
                ]);
            } else {
                // Insert new comprehensive joining details
                $joining_query = "INSERT INTO joining_details 
                                 (candidate_id, expected_joining_date, actual_joining_date, salary_offered, 
                                  salary_negotiated, position, department, reporting_manager, work_location,
                                  employment_type, joining_notes, joining_feedback, documents_required,
                                  probation_period_months, joining_status, created_at) 
                                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'joined', NOW())";
                $stmt = $db->prepare($joining_query);
                $stmt->execute([
                    $candidate_id, $joining_date, $actual_joining_date, $salary_offered,
                    $salary_negotiated, $position, $department, $reporting_manager, $work_location,
                    $employment_type, $joining_notes, $joining_feedback, $documents_required,
                    $probation_period
                ]);
            }
            
            // Insert timeline record
            insertTimelineRecord($db, $candidate_id, null, [
                'step_number' => 6,
                'step_name' => 'Candidate Joined',
                'step_type' => 'joining',
                'step_status' => 'completed',
                'data' => [
                    'expected_joining_date' => $joining_date,
                    'actual_joining_date' => $actual_joining_date,
                    'position' => $position,
                    'department' => $department,
                    'salary_offered' => $salary_offered,
                    'employment_type' => $employment_type
                ],
                'performed_by' => $user_id,
                'performed_by_name' => $username,
                'notes' => "Candidate successfully joined the organization"
            ]);
            
            // Update process completion
            $process_query = "INSERT INTO process_completion 
                             (candidate_id, process_status, completion_reason, final_result, 
                              started_date, completed_date, completed_by, completed_by_name, process_notes) 
                             VALUES (?, 'completed', 'successful_hire', 'hired', ?, NOW(), ?, ?, ?)
                             ON DUPLICATE KEY UPDATE 
                             process_status = 'completed', completion_reason = 'successful_hire', final_result = 'hired',
                             completed_date = NOW(), completed_by = ?, completed_by_name = ?, process_notes = ?";
            $stmt = $db->prepare($process_query);
            $stmt->execute([
                $candidate_id, $candidate['added_date'], $user_id, $username, $joining_feedback,
                $user_id, $username, $joining_feedback
            ]);
            
            // Insert status history
            insertStatusHistory($db, $candidate_id, $candidate['status'], 'joined', $user_id, 
                "Candidate joined successfully on " . ($actual_joining_date ?: $joining_date), 
                [
                    'expected_joining_date' => $joining_date,
                    'actual_joining_date' => $actual_joining_date,
                    'position' => $position,
                    'department' => $department,
                    'salary_offered' => $salary_offered,
                    'salary_negotiated' => $salary_negotiated,
                    'employment_type' => $employment_type
                ]
            );
            
            $success_message = "Candidate joining confirmed successfully";
            break;
            
        case 'not_joined':
            // Get comprehensive joining details
            $joining_date = $_POST['joining_date'] ?? '';
            $salary_offered = $_POST['salary_offered'] ?? null;
            $position = $_POST['position'] ?? '';
            $department = $_POST['department'] ?? '';
            $joining_notes = $_POST['joining_notes'] ?? '';
            $joining_feedback = $_POST['joining_feedback'] ?? '';
            $not_joining_reason = $_POST['not_joining_reason'] ?? '';
            
            // Update candidate status
            $update_query = "UPDATE candidates SET status = 'not_joined', updated_at = NOW() WHERE id = ?";
            $stmt = $db->prepare($update_query);
            $stmt->execute([$candidate_id]);
            
            // Insert or update joining details
            $joining_check_query = "SELECT id FROM joining_details WHERE candidate_id = ?";
            $stmt = $db->prepare($joining_check_query);
            $stmt->execute([$candidate_id]);
            $existing_joining = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existing_joining) {
                // Update existing joining details
                $joining_update_query = "UPDATE joining_details SET 
                                        joining_status = 'not_joined', 
                                        joining_feedback = ?, 
                                        updated_at = NOW() 
                                        WHERE candidate_id = ?";
                $stmt = $db->prepare($joining_update_query);
                $stmt->execute([$joining_feedback, $candidate_id]);
            } else {
                // Insert new joining details
                $joining_query = "INSERT INTO joining_details 
                                 (candidate_id, expected_joining_date, salary_offered, position, department, 
                                  joining_notes, joining_feedback, joining_status, created_at) 
                                 VALUES (?, ?, ?, ?, ?, ?, ?, 'not_joined', NOW())";
                $stmt = $db->prepare($joining_query);
                $stmt->execute([
                    $candidate_id, $joining_date, $salary_offered, $position, $department,
                    $joining_notes, $joining_feedback
                ]);
            }
            
            // Insert timeline record
            insertTimelineRecord($db, $candidate_id, null, [
                'step_number' => 6,
                'step_name' => 'Candidate Did Not Join',
                'step_type' => 'joining',
                'step_status' => 'completed',
                'data' => [
                    'expected_joining_date' => $joining_date,
                    'position' => $position,
                    'department' => $department,
                    'not_joining_reason' => $not_joining_reason,
                    'feedback' => $joining_feedback
                ],
                'performed_by' => $user_id,
                'performed_by_name' => $username,
                'notes' => "Candidate was selected but did not join"
            ]);
            
            // Update process completion
            $process_query = "INSERT INTO process_completion 
                             (candidate_id, process_status, completion_reason, final_result, 
                              started_date, completed_date, completed_by, completed_by_name, process_notes) 
                             VALUES (?, 'completed', 'candidate_no_show', 'not_joined', ?, NOW(), ?, ?, ?)
                             ON DUPLICATE KEY UPDATE 
                             process_status = 'completed', completion_reason = 'candidate_no_show', final_result = 'not_joined',
                             completed_date = NOW(), completed_by = ?, completed_by_name = ?, process_notes = ?";
            $stmt = $db->prepare($process_query);
            $stmt->execute([
                $candidate_id, $candidate['added_date'], $user_id, $username, $joining_feedback,
                $user_id, $username, $joining_feedback
            ]);
            
            // Insert status history
            insertStatusHistory($db, $candidate_id, $candidate['status'], 'not_joined', $user_id, 
                "Candidate was selected but did not join", 
                [
                    'expected_joining_date' => $joining_date,
                    'position' => $position,
                    'department' => $department,
                    'salary_offered' => $salary_offered,
                    'reason' => $joining_feedback
                ]
            );
            
            $success_message = "Candidate not joined status updated";
            break;
            
        default:
            throw new Exception('Invalid process action: ' . $process_action);
    }
    
    // Ensure transaction is committed
    $db->commit();
    $_SESSION['success'] = $success_message;
    error_log("Transaction committed successfully for candidate ID: $candidate_id, Action: $process_action");
    
} catch (Exception $e) {
    $db->rollback();
    $_SESSION['error'] = 'Error processing interview: ' . $e->getMessage();
    error_log("Interview Process Error: " . $e->getMessage());
}

// Clean output buffer and redirect
ob_end_clean();
header('Location: ' . $redirect_url);
exit();
?>
