<?php
session_start();

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.html');
    exit();
}

include_once 'config/database.php';
include_once 'classes/Candidate.php';

$database = new Database();
$db = $database->getConnection();

// Get all users for filter dropdown
$users_query = "SELECT id, username FROM users ORDER BY username";
$users_stmt = $db->prepare($users_query);
$users_stmt->execute();
$users = $users_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get all sources for filter dropdown
$sources_query = "SELECT DISTINCT source FROM candidates WHERE source IS NOT NULL AND source != '' ORDER BY source";
$sources_stmt = $db->prepare($sources_query);
$sources_stmt->execute();
$sources = $sources_stmt->fetchAll(PDO::FETCH_ASSOC);

// Handle search and filters
$search_term = $_GET['search'] ?? '';
$status_filter = $_GET['status'] ?? '';
$source_filter = $_GET['source'] ?? '';
$user_filter = $_GET['user_filter'] ?? '';
$date_filter_type = $_GET['date_filter_type'] ?? 'single';
$lineup_date = $_GET['lineup_date'] ?? '';
$lineup_date_from = $_GET['lineup_date_from'] ?? '';
$lineup_date_to = $_GET['lineup_date_to'] ?? '';
$page = (int)($_GET['page'] ?? 1);
$per_page = 20;
$offset = ($page - 1) * $per_page;

// Build the WHERE clause
$where_conditions = [];
$params = [];

if (!empty($search_term)) {
    $where_conditions[] = "(c.name LIKE ? OR c.candidate_id LIKE ? OR c.phone LIKE ? OR c.email LIKE ?)";
    $search_param = "%$search_term%";
    $params = array_merge($params, [$search_param, $search_param, $search_param, $search_param]);
}

if (!empty($status_filter)) {
    $where_conditions[] = "c.status = ?";
    $params[] = $status_filter;
}

if (!empty($source_filter)) {
    $where_conditions[] = "c.source = ?";
    $params[] = $source_filter;
}

if (!empty($user_filter)) {
    $where_conditions[] = "c.added_by = ?";
    $params[] = $user_filter;
}

// Date filtering
if ($date_filter_type === 'single' && !empty($lineup_date)) {
    $where_conditions[] = "DATE(c.lineup_date) = ?";
    $params[] = $lineup_date;
} elseif ($date_filter_type === 'range' && !empty($lineup_date_from) && !empty($lineup_date_to)) {
    $where_conditions[] = "DATE(c.lineup_date) BETWEEN ? AND ?";
    $params[] = $lineup_date_from;
    $params[] = $lineup_date_to;
}

$where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

// Get total count for pagination
$count_query = "SELECT COUNT(*) as total FROM candidates c $where_clause";
$count_stmt = $db->prepare($count_query);
$count_stmt->execute($params);
$total_records = $count_stmt->fetch(PDO::FETCH_ASSOC)['total'];
$total_pages = ceil($total_records / $per_page);

// Get candidates with user information
$query = "SELECT c.*, u.username as added_by_name 
          FROM candidates c 
          LEFT JOIN users u ON c.added_by = u.id 
          $where_clause 
          ORDER BY c.lineup_date DESC, c.added_date DESC 
          LIMIT $per_page OFFSET $offset";

$stmt = $db->prepare($query);
$stmt->execute($params);
$candidates = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get statistics
$stats_query = "SELECT 
    COUNT(*) as total,
    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
    SUM(CASE WHEN status = 'shortlisted' THEN 1 ELSE 0 END) as shortlisted,
    SUM(CASE WHEN status = 'selected' THEN 1 ELSE 0 END) as selected,
    SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected,
    SUM(CASE WHEN status = 'joined' THEN 1 ELSE 0 END) as joined,
    SUM(CASE WHEN status = 'hold' THEN 1 ELSE 0 END) as on_hold
    FROM candidates c $where_clause";

$stats_stmt = $db->prepare($stats_query);
$stats_stmt->execute($params);
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Search Candidates - HR Portal</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .candidate-card {
            transition: all 0.3s ease;
            border-left: 4px solid #dee2e6;
        }
        .candidate-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }
        .candidate-card.status-pending { border-left-color: #ffc107; }
        .candidate-card.status-shortlisted { border-left-color: #17a2b8; }
        .candidate-card.status-selected { border-left-color: #28a745; }
        .candidate-card.status-rejected { border-left-color: #dc3545; }
        .candidate-card.status-joined { border-left-color: #007bff; }
        .candidate-card.status-hold { border-left-color: #6c757d; }
        
        .stats-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .filter-section {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .date-toggle {
            display: flex;
            background: #e9ecef;
            border-radius: 25px;
            padding: 3px;
            margin-bottom: 15px;
        }
        
        .date-toggle button {
            flex: 1;
            border: none;
            background: transparent;
            padding: 8px 16px;
            border-radius: 20px;
            transition: all 0.3s ease;
        }
        
        .date-toggle button.active {
            background: #007bff;
            color: white;
        }
        
        .search-results-header {
            background: white;
            padding: 15px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
        
        .action-buttons {
            display: flex;
            gap: 5px;
        }
        
        .btn-sm {
            padding: 0.25rem 0.5rem;
            font-size: 0.875rem;
        }
        
        .pagination-wrapper {
            display: flex;
            justify-content: between;
            align-items: center;
            margin-top: 20px;
        }
        
        .results-info {
            color: #6c757d;
            font-size: 0.9rem;
        }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="dashboard.php">
                <i class="fas fa-users me-2"></i>HR Portal
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="dashboard.php">
                    <i class="fas fa-dashboard me-1"></i>Dashboard
                </a>
                <a class="nav-link" href="add-candidate.html">
                    <i class="fas fa-plus me-1"></i>Add Candidate
                </a>
                <a class="nav-link" href="logout.php">
                    <i class="fas fa-sign-out-alt me-1"></i>Logout
                </a>
            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <!-- Statistics Cards -->
        <div class="row mb-4">
            <div class="col-md-12">
                <div class="stats-card p-4 rounded">
                    <h5 class="mb-3"><i class="fas fa-chart-bar me-2"></i>Search Results Statistics</h5>
                    <div class="row text-center">
                        <div class="col-md-2">
                            <h3><?php echo $stats['total']; ?></h3>
                            <small>Total</small>
                        </div>
                        <div class="col-md-2">
                            <h3><?php echo $stats['pending']; ?></h3>
                            <small>Pending</small>
                        </div>
                        <div class="col-md-2">
                            <h3><?php echo $stats['shortlisted']; ?></h3>
                            <small>Shortlisted</small>
                        </div>
                        <div class="col-md-2">
                            <h3><?php echo $stats['selected']; ?></h3>
                            <small>Selected</small>
                        </div>
                        <div class="col-md-2">
                            <h3><?php echo $stats['rejected']; ?></h3>
                            <small>Rejected</small>
                        </div>
                        <div class="col-md-2">
                            <h3><?php echo $stats['joined']; ?></h3>
                            <small>Joined</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Search and Filter Section -->
        <div class="filter-section">
            <form method="GET" id="searchForm">
                <div class="row">
                    <div class="col-md-4">
                        <label class="form-label">Search Candidates</label>
                        <div class="input-group">
                            <input type="text" class="form-control" name="search" 
                                   value="<?php echo htmlspecialchars($search_term); ?>" 
                                   placeholder="Name, ID, Phone, Email...">
                            <button class="btn btn-primary" type="submit">
                                <i class="fas fa-search"></i>
                            </button>
                        </div>
                    </div>
                    
                    <div class="col-md-2">
                        <label class="form-label">Status</label>
                        <select class="form-select" name="status" onchange="this.form.submit()">
                            <option value="">All Status</option>
                            <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                            <option value="shortlisted" <?php echo $status_filter === 'shortlisted' ? 'selected' : ''; ?>>Shortlisted</option>
                            <option value="selected" <?php echo $status_filter === 'selected' ? 'selected' : ''; ?>>Selected</option>
                            <option value="rejected" <?php echo $status_filter === 'rejected' ? 'selected' : ''; ?>>Rejected</option>
                            <option value="joined" <?php echo $status_filter === 'joined' ? 'selected' : ''; ?>>Joined</option>
                            <option value="hold" <?php echo $status_filter === 'hold' ? 'selected' : ''; ?>>On Hold</option>
                        </select>
                    </div>
                    
                    <div class="col-md-2">
                        <label class="form-label">Source</label>
                        <select class="form-select" name="source" onchange="this.form.submit()">
                            <option value="">All Sources</option>
                            <?php foreach ($sources as $source): ?>
                                <option value="<?php echo htmlspecialchars($source['source']); ?>" 
                                        <?php echo $source_filter === $source['source'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($source['source']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="col-md-2">
                        <label class="form-label">Added By</label>
                        <select class="form-select" name="user_filter" onchange="this.form.submit()">
                            <option value="">All Users</option>
                            <?php foreach ($users as $user): ?>
                                <option value="<?php echo $user['id']; ?>" 
                                        <?php echo $user_filter == $user['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($user['username']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="col-md-2">
                        <label class="form-label">Actions</label>
                        <div class="d-grid">
                            <button type="button" class="btn btn-success" onclick="exportResults()">
                                <i class="fas fa-download me-1"></i>Export
                            </button>
                        </div>
                    </div>
                </div>
                
                <!-- Date Filter Section -->
                <div class="row mt-3">
                    <div class="col-md-6">
                        <label class="form-label">Lineup Date Filter</label>
                        
                        <!-- Date Filter Type Toggle -->
                        <div class="date-toggle">
                            <button type="button" class="<?php echo $date_filter_type === 'single' ? 'active' : ''; ?>" 
                                    onclick="toggleDateFilter('single')">
                                <i class="fas fa-calendar-day me-1"></i>Single Date
                            </button>
                            <button type="button" class="<?php echo $date_filter_type === 'range' ? 'active' : ''; ?>" 
                                    onclick="toggleDateFilter('range')">
                                <i class="fas fa-calendar-week me-1"></i>Date Range
                            </button>
                        </div>
                        
                        <input type="hidden" name="date_filter_type" id="date_filter_type" value="<?php echo $date_filter_type; ?>">
                        
                        <!-- Single Date Filter -->
                        <div id="single_date_filter" style="<?php echo $date_filter_type === 'single' ? '' : 'display: none;'; ?>">
                            <input type="date" class="form-control" name="lineup_date" 
                                   value="<?php echo htmlspecialchars($lineup_date); ?>" 
                                   onchange="this.form.submit()">
                        </div>
                        
                        <!-- Date Range Filter -->
                        <div id="range_date_filter" style="<?php echo $date_filter_type === 'range' ? '' : 'display: none;'; ?>">
                            <div class="row">
                                <div class="col-6">
                                    <input type="date" class="form-control" name="lineup_date_from" 
                                           value="<?php echo htmlspecialchars($lineup_date_from); ?>" 
                                           placeholder="From Date" onchange="this.form.submit()">
                                </div>
                                <div class="col-6">
                                    <input type="date" class="form-control" name="lineup_date_to" 
                                           value="<?php echo htmlspecialchars($lineup_date_to); ?>" 
                                           placeholder="To Date" onchange="this.form.submit()">
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-6 d-flex align-items-end">
                        <button type="button" class="btn btn-secondary me-2" onclick="clearFilters()">
                            <i class="fas fa-times me-1"></i>Clear Filters
                        </button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-filter me-1"></i>Apply Filters
                        </button>
                    </div>
                </div>
            </form>
        </div>

        <!-- Search Results Header -->
        <div class="search-results-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h5 class="mb-0">
                        <i class="fas fa-users me-2"></i>Search Results
                        <?php if (!empty($search_term) || !empty($status_filter) || !empty($source_filter) || !empty($user_filter) || !empty($lineup_date) || (!empty($lineup_date_from) && !empty($lineup_date_to))): ?>
                            <small class="text-muted">(Filtered)</small>
                        <?php endif; ?>
                    </h5>
                    <div class="results-info">
                        Showing <?php echo $offset + 1; ?>-<?php echo min($offset + $per_page, $total_records); ?> 
                        of <?php echo $total_records; ?> candidates
                    </div>
                </div>
                <div>
                    <?php if (!empty($user_filter)): ?>
                        <?php 
                        $selected_user = array_filter($users, function($u) use ($user_filter) { 
                            return $u['id'] == $user_filter; 
                        });
                        $selected_user = reset($selected_user);
                        ?>
                        <span class="badge bg-info">
                            <i class="fas fa-user me-1"></i>
                            Showing entries by: <?php echo htmlspecialchars($selected_user['username']); ?>
                        </span>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Candidates List -->
        <div class="row">
            <?php if (empty($candidates)): ?>
                <div class="col-12">
                    <div class="text-center py-5">
                        <i class="fas fa-search fa-3x text-muted mb-3"></i>
                        <h4 class="text-muted">No candidates found</h4>
                        <p class="text-muted">Try adjusting your search criteria or filters</p>
                        <button class="btn btn-primary" onclick="clearFilters()">
                            <i class="fas fa-times me-1"></i>Clear All Filters
                        </button>
                    </div>
                </div>
            <?php else: ?>
                <?php foreach ($candidates as $candidate): ?>
                    <div class="col-md-6 col-lg-4 mb-3">
                        <div class="card candidate-card status-<?php echo $candidate['status']; ?> h-100">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-start mb-2">
                                    <h6 class="card-title mb-0">
                                        <?php echo htmlspecialchars($candidate['name']); ?>
                                    </h6>
                                    <span class="badge bg-<?php 
                                        echo $candidate['status'] === 'pending' ? 'warning' : 
                                            ($candidate['status'] === 'selected' ? 'success' : 
                                            ($candidate['status'] === 'rejected' ? 'danger' : 
                                            ($candidate['status'] === 'hold' ? 'secondary' :
                                            ($candidate['status'] === 'shortlisted' ? 'info' : 
                                            ($candidate['status'] === 'joined' ? 'primary' : 'secondary'))))); 
                                    ?>">
                                        <?php echo ucfirst(str_replace('_', ' ', $candidate['status'])); ?>
                                    </span>
                                </div>
                                
                                <div class="candidate-info mb-3">
                                    <small class="text-muted d-block">
                                        <i class="fas fa-id-badge me-1"></i>
                                        ID: <?php echo htmlspecialchars($candidate['candidate_id']); ?>
                                    </small>
                                    <small class="text-muted d-block">
                                        <i class="fas fa-phone me-1"></i>
                                        <?php echo htmlspecialchars($candidate['phone']); ?>
                                    </small>
                                    <small class="text-muted d-block">
                                        <i class="fas fa-envelope me-1"></i>
                                        <?php echo htmlspecialchars($candidate['email']); ?>
                                    </small>
                                    <small class="text-muted d-block">
                                        <i class="fas fa-calendar me-1"></i>
                                        Lineup: <?php echo date('d-M-Y', strtotime($candidate['lineup_date'])); ?>
                                    </small>
                                    <?php if ($candidate['source']): ?>
                                    <small class="text-muted d-block">
                                        <i class="fas fa-source me-1"></i>
                                        Source: <?php echo htmlspecialchars($candidate['source']); ?>
                                    </small>
                                    <?php endif; ?>
                                    <small class="text-muted d-block">
                                        <i class="fas fa-user-plus me-1"></i>
                                        Added by: <?php echo htmlspecialchars($candidate['added_by_name'] ?? 'Unknown'); ?>
                                    </small>
                                </div>
                                
                                <div class="action-buttons">
                                    <button class="btn btn-primary btn-sm" onclick="viewCandidate('<?php echo $candidate['candidate_id']; ?>')">
                                        <i class="fas fa-eye"></i> View
                                    </button>
                                    <button class="btn btn-success btn-sm" onclick="startInterview('<?php echo $candidate['candidate_id']; ?>')">
                                        <i class="fas fa-play"></i> Interview
                                    </button>
                                    <button class="btn btn-info btn-sm" onclick="editCandidate('<?php echo $candidate['candidate_id']; ?>')">
                                        <i class="fas fa-edit"></i> Edit
                                    </button>
                                    <button class="btn btn-secondary btn-sm" onclick="viewHistory('<?php echo $candidate['candidate_id']; ?>')">
                                        <i class="fas fa-history"></i> History
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <!-- Pagination -->
        <?php if ($total_pages > 1): ?>
        <div class="pagination-wrapper">
            <div class="results-info">
                Page <?php echo $page; ?> of <?php echo $total_pages; ?>
            </div>
            <nav>
                <ul class="pagination">
                    <?php if ($page > 1): ?>
                        <li class="page-item">
                            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>">
                                <i class="fas fa-chevron-left"></i>
                            </a>
                        </li>
                    <?php endif; ?>
                    
                    <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                        <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>">
                                <?php echo $i; ?>
                            </a>
                        </li>
                    <?php endfor; ?>
                    
                    <?php if ($page < $total_pages): ?>
                        <li class="page-item">
                            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>">
                                <i class="fas fa-chevron-right"></i>
                            </a>
                        </li>
                    <?php endif; ?>
                </ul>
            </nav>
        </div>
        <?php endif; ?>
    </div>

    <!-- Interview Process Modal -->
    <div class="modal fade" id="interviewModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-clipboard-list me-2"></i>Interview Process
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="interviewModalBody">
                    <!-- Interview process content will be loaded here -->
                </div>
            </div>
        </div>
    </div>

    <!-- View Candidate Modal -->
    <div class="modal fade" id="viewModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-user me-2"></i>Candidate Details
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="viewModalBody">
                    <!-- Candidate details will be loaded here -->
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function toggleDateFilter(type) {
            document.getElementById('date_filter_type').value = type;
            
            // Update toggle buttons
            document.querySelectorAll('.date-toggle button').forEach(btn => {
                btn.classList.remove('active');
            });
            event.target.classList.add('active');
            
            // Show/hide appropriate filter
            if (type === 'single') {
                document.getElementById('single_date_filter').style.display = 'block';
                document.getElementById('range_date_filter').style.display = 'none';
                // Clear range values
                document.querySelector('input[name="lineup_date_from"]').value = '';
                document.querySelector('input[name="lineup_date_to"]').value = '';
            } else {
                document.getElementById('single_date_filter').style.display = 'none';
                document.getElementById('range_date_filter').style.display = 'block';
                // Clear single date value
                document.querySelector('input[name="lineup_date"]').value = '';
            }
            
            // Submit form to apply filter
            document.getElementById('searchForm').submit();
        }
        
        function clearFilters() {
            // Clear all form inputs
            document.querySelector('input[name="search"]').value = '';
            document.querySelector('select[name="status"]').value = '';
            document.querySelector('select[name="source"]').value = '';
            document.querySelector('select[name="user_filter"]').value = '';
            document.querySelector('input[name="lineup_date"]').value = '';
            document.querySelector('input[name="lineup_date_from"]').value = '';
            document.querySelector('input[name="lineup_date_to"]').value = '';
            
            // Reset to single date filter
            document.getElementById('date_filter_type').value = 'single';
            document.getElementById('single_date_filter').style.display = 'block';
            document.getElementById('range_date_filter').style.display = 'none';
            
            // Update toggle buttons
            document.querySelectorAll('.date-toggle button').forEach(btn => {
                btn.classList.remove('active');
            });
            document.querySelector('.date-toggle button:first-child').classList.add('active');
            
            // Submit form
            document.getElementById('searchForm').submit();
        }
        
        function viewCandidate(candidateId) {
            fetch(`ajax/view-candidate.php?candidate_id=${candidateId}`)
                .then(response => response.text())
                .then(data => {
                    document.getElementById('viewModalBody').innerHTML = data;
                    new bootstrap.Modal(document.getElementById('viewModal')).show();
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error loading candidate details');
                });
        }
        
        function startInterview(candidateId) {
            fetch(`ajax/interview-process-simple.php?candidate_id=${candidateId}`)
                .then(response => response.text())
                .then(data => {
                    document.getElementById('interviewModalBody').innerHTML = data;
                    new bootstrap.Modal(document.getElementById('interviewModal')).show();
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error loading interview process');
                });
        }
        
        function editCandidate(candidateId) {
            window.open(`edit-candidate.html?candidate_id=${candidateId}`, '_blank');
        }
        
        function viewHistory(candidateId) {
            window.open(`interview-process-history.html?candidate_id=${candidateId}`, '_blank');
        }
        
        function exportResults() {
            // Get current search parameters
            const params = new URLSearchParams(window.location.search);
            params.set('export', '1');
            
            // Create download link
            const downloadUrl = `ajax/export-candidates.php?${params.toString()}`;
            window.open(downloadUrl, '_blank');
        }
        
        // Auto-submit form on input changes with debouncing
        let searchTimeout;
        document.querySelector('input[name="search"]').addEventListener('input', function() {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(() => {
                this.form.submit();
            }, 500);
        });
        
        // Show loading state for buttons
        document.addEventListener('click', function(e) {
            if (e.target.matches('.btn-primary, .btn-success, .btn-info, .btn-secondary')) {
                const btn = e.target;
                const originalText = btn.innerHTML;
                btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Loading...';
                btn.disabled = true;
                
                // Re-enable after 3 seconds (fallback)
                setTimeout(() => {
                    btn.innerHTML = originalText;
                    btn.disabled = false;
                }, 3000);
            }
        });
    </script>
</body>
</html>
