<?php
include_once 'config/database.php';
include_once 'includes/session.php';

// Check if user is logged in
if (!isLoggedIn()) {
    header('Location: login.html');
    exit();
}

$database = new Database();
$db = $database->getConnection();

$candidate_id = $_GET['candidate_id'] ?? '';
if (empty($candidate_id)) {
    $_SESSION['error'] = 'Candidate ID is required';
    header('Location: search-candidates.php');
    exit();
}

// Get candidate details
$candidate_query = "SELECT c.*, s.name as source_name, p.name as profile_name 
                   FROM candidates c 
                   LEFT JOIN sources s ON c.source_id = s.id 
                   LEFT JOIN profiles p ON c.profile_id = p.id 
                   WHERE c.candidate_id = ?";
$stmt = $db->prepare($candidate_query);
$stmt->execute([$candidate_id]);
$candidate = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$candidate) {
    $_SESSION['error'] = 'Candidate not found';
    header('Location: search-candidates.php');
    exit();
}

// Get comprehensive interview history
$interview_query = "SELECT i.*, 
                           CASE 
                               WHEN i.interviewer_name IS NOT NULL THEN i.interviewer_name
                               WHEN u.username IS NOT NULL THEN u.username
                               ELSE 'N/A'
                           END as interviewer_display_name
                    FROM interviews i 
                    LEFT JOIN users u ON i.interviewer_id = u.id
                    WHERE i.candidate_id = ? 
                    ORDER BY i.interview_date DESC, i.created_at DESC";
$stmt = $db->prepare($interview_query);
$stmt->execute([$candidate['id']]);
$interviews = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get interview feedback
$feedback_query = "SELECT f.*, u.username as interviewer_username
                   FROM interview_feedback f
                   LEFT JOIN users u ON f.interviewer_id = u.id
                   WHERE f.candidate_id = ?
                   ORDER BY f.feedback_date DESC";
$stmt = $db->prepare($feedback_query);
$stmt->execute([$candidate['id']]);
$feedback_records = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get joining details
$joining_query = "SELECT * FROM joining_details WHERE candidate_id = ? ORDER BY created_at DESC LIMIT 1";
$stmt = $db->prepare($joining_query);
$stmt->execute([$candidate['id']]);
$joining_details = $stmt->fetch(PDO::FETCH_ASSOC);

// Get reschedule history
$reschedule_query = "SELECT * FROM reschedules WHERE candidate_id = ? ORDER BY created_at DESC";
$stmt = $db->prepare($reschedule_query);
$stmt->execute([$candidate['id']]);
$reschedules = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get status history
$status_query = "SELECT * FROM candidate_status_history WHERE candidate_id = ? ORDER BY changed_at DESC";
$stmt = $db->prepare($status_query);
$stmt->execute([$candidate['id']]);
$status_history = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get process timeline
$timeline_query = "SELECT * FROM interview_process_timeline WHERE candidate_id = ? ORDER BY step_number ASC, created_at ASC";
$stmt = $db->prepare($timeline_query);
$stmt->execute([$candidate['id']]);
$timeline = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get process completion details
$process_query = "SELECT * FROM process_completion WHERE candidate_id = ? ORDER BY created_at DESC LIMIT 1";
$stmt = $db->prepare($process_query);
$stmt->execute([$candidate['id']]);
$process_completion = $stmt->fetch(PDO::FETCH_ASSOC);

// Get offers
$offers_query = "SELECT * FROM offers WHERE candidate_id = ? ORDER BY created_at DESC";
$stmt = $db->prepare($offers_query);
$stmt->execute([$candidate['id']]);
$offers = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Detailed Candidate View - <?php echo htmlspecialchars($candidate['name']); ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background-color: #f8f9fa;
        }
        .navbar {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        .detail-container {
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            padding: 30px;
            margin: 20px 0;
        }
        .status-badge {
            font0,0,0.1);
            padding: 30px;
            margin: 20px 0;
        }
        .status-badge {
            font-size: 1rem;
            padding: 8px 15px;
            border-radius: 20px;
            font-weight: bold;
        }
        .status-pending { background-color: #fff3cd; color: #856404; }
        .status-shortlisted { background-color: #d4edda; color: #155724; }
        .status-selected { background-color: #d1ecf1; color: #0c5460; }
        .status-rejected { background-color: #f8d7da; color: #721c24; }
        .status-hold { background-color: #e2e3e5; color: #383d41; }
        .status-joined { background-color: #d1ecf1; color: #0c5460; }
        .status-not_joined { background-color: #f8d7da; color: #721c24; }
        .status-process_ended { background-color: #e2e3e5; color: #383d41; }
        .status-offer_declined { background-color: #f8d7da; color: #721c24; }
        .status-rescheduled { background-color: #fff3cd; color: #856404; }
        
        .info-card {
            background-color: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            border-left: 4px solid #007bff;
        }
        
        .timeline {
            position: relative;
            padding-left: 30px;
        }
        .timeline::before {
            content: '';
            position: absolute;
            left: 15px;
            top: 0;
            bottom: 0;
            width: 2px;
            background: #dee2e6;
        }
        .timeline-item {
            position: relative;
            margin-bottom: 20px;
            background: white;
            padding: 15px;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .timeline-item::before {
            content: '';
            position: absolute;
            left: -23px;
            top: 20px;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background: #007bff;
        }
        
        .score-badge {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 0.8rem;
            font-weight: bold;
            margin: 2px;
        }
        .score-excellent { background-color: #d4edda; color: #155724; }
        .score-good { background-color: #d1ecf1; color: #0c5460; }
        .score-average { background-color: #fff3cd; color: #856404; }
        .score-poor { background-color: #f8d7da; color: #721c24; }
        
        .section-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 15px 20px;
            border-radius: 10px 10px 0 0;
            margin: -20px -20px 20px -20px;
        }
        
        .data-table {
            font-size: 0.9rem;
        }
        .data-table th {
            background-color: #f8f9fa;
            font-weight: 600;
            border-top: none;
        }
        
        @media print {
            .no-print { display: none !important; }
            body { background: white !important; }
            .detail-container { box-shadow: none !important; }
        }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark no-print">
        <div class="container">
            <a class="navbar-brand" href="dashboard.php">
                <i class="fas fa-users me-2"></i>
                Candidate Management System
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="search-candidates.php">
                    <i class="fas fa-search me-1"></i>Back to Search
                </a>
                <button class="btn btn-outline-light btn-sm ms-2" onclick="logout()">
                    <i class="fas fa-sign-out-alt me-1"></i>Logout
                </button>
            </div>
        </div>
    </nav>

    <div class="container">
        <div class="detail-container">
            <div class="d-flex justify-content-between align-items-center mb-4 no-print">
                <h2><i class="fas fa-user-circle me-2"></i>Comprehensive Candidate Profile</h2>
                <div>
                    <button class="btn btn-outline-primary me-2" onclick="window.print()">
                        <i class="fas fa-print me-1"></i>Print Report
                    </button>
                    <button class="btn btn-primary" onclick="editCandidate()">
                        <i class="fas fa-edit me-1"></i>Edit Details
                    </button>
                </div>
            </div>

            <!-- Candidate Basic Information -->
            <div class="info-card">
                <div class="section-header">
                    <h5 class="mb-0"><i class="fas fa-id-card me-2"></i>Basic Information</h5>
                </div>
                <div class="row">
                    <div class="col-md-8">
                        <h3 class="mb-3">
                            <?php echo htmlspecialchars($candidate['name']); ?>
                            <span class="status-badge status-<?php echo $candidate['status']; ?>">
                                <?php echo strtoupper(str_replace('_', ' ', $candidate['status'])); ?>
                            </span>
                        </h3>
                        <div class="row">
                            <div class="col-md-6">
                                <p><strong><i class="fas fa-id-badge me-2"></i>Candidate ID:</strong> <?php echo $candidate['candidate_id']; ?></p>
                                <p><strong><i class="fas fa-envelope me-2"></i>Email:</strong> <a href="mailto:<?php echo $candidate['email']; ?>"><?php echo $candidate['email']; ?></a></p>
                                <p><strong><i class="fas fa-phone me-2"></i>Phone:</strong> <a href="tel:<?php echo $candidate['phone']; ?>"><?php echo $candidate['phone']; ?></a></p>
                            </div>
                             <div class="col-md-6">
                                <p><strong><i class="fas fa-phone me-2"></i>Location:</strong> <a href="tel:<?php echo $candidate['location']; ?>"><?php echo $candidate['location']; ?></a></p>
                                <p><strong><i class="fas fa-phone me-2"></i>Team Name:</strong> <a href="tel:<?php echo $candidate['team_name']; ?>"><?php echo $candidate['team_name']; ?></a></p>
                            </div>
                            <div class="col-md-6">
                                <p><strong><i class="fas fa-calendar-plus me-2"></i>Added Date:</strong> <?php echo date('d-M-Y', strtotime($candidate['added_date'])); ?></p>
                                <p><strong><i class="fas fa-calendar me-2"></i>Lineup Date:</strong> <?php echo date('d-M-Y', strtotime($candidate['lineup_date'])); ?></p>
                                <p><strong><i class="fas fa-clock me-2"></i>Last Updated:</strong> <?php echo $candidate['updated_at'] ? date('d-M-Y H:i', strtotime($candidate['updated_at'])) : 'N/A'; ?></p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="text-center">
                            <div class="bg-light p-3 rounded">
                                <p><strong><i class="fas fa-briefcase me-2"></i>Profile:</strong><br><?php echo htmlspecialchars($candidate['profile_name'] ?? 'N/A'); ?></p>
                                <p><strong><i class="fas fa-source me-2"></i>Source:</strong><br><?php echo htmlspecialchars($candidate['source_name'] ?? 'N/A'); ?></p>
                                <?php if (!empty($candidate['referral_code'])): ?>
                                <p><strong><i class="fas fa-code me-2"></i>Referral Code:</strong><br><?php echo htmlspecialchars($candidate['referral_code']); ?></p>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Process Overview -->
            <?php if ($process_completion): ?>
            <div class="info-card">
                <div class="section-header">
                    <h5 class="mb-0"><i class="fas fa-chart-line me-2"></i>Process Overview</h5>
                </div>
                <div class="row">
                    <div class="col-md-3">
                        <div class="text-center p-3 bg-light rounded">
                            <h4 class="text-primary"><?php echo $process_completion['total_interviews']; ?></h4>
                            <small>Total Interviews</small>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="text-center p-3 bg-light rounded">
                            <h4 class="text-warning"><?php echo $process_completion['total_reschedules']; ?></h4>
                            <small>Reschedules</small>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="text-center p-3 bg-light rounded">
                            <h4 class="text-info"><?php echo $process_completion['process_duration_days'] ?? 'N/A'; ?></h4>
                            <small>Process Days</small>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="text-center p-3 bg-light rounded">
                            <h4 class="text-<?php echo $process_completion['final_result'] === 'hired' ? 'success' : ($process_completion['final_result'] === 'rejected' ? 'danger' : 'secondary'); ?>">
                                <?php echo strtoupper(str_replace('_', ' ', $process_completion['final_result'] ?? 'ACTIVE')); ?>
                            </h4>
                            <small>Final Result</small>
                        </div>
                    </div>
                </div>
                <?php if ($process_completion['process_notes']): ?>
                <div class="mt-3">
                    <strong>Process Notes:</strong>
                    <p class="text-muted"><?php echo nl2br(htmlspecialchars($process_completion['process_notes'])); ?></p>
                </div>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <!-- Interview History with Detailed Information -->
            <?php if (!empty($interviews)): ?>
            <div class="info-card">
                <div class="section-header">
                    <h5 class="mb-0"><i class="fas fa-clipboard-list me-2"></i>Complete Interview History</h5>
                </div>
                <div class="table-responsive">
                    <table class="table table-sm data-table">
                        <thead>
                            <tr>
                                <th>Date & Time</th>
                                <th>Type</th>
                                <th>Interviewer</th>
                                <th>Duration</th>
                                <th>Turn Up</th>
                                <th>Result</th>
                                <th>Scores</th>
                                <th>Notes</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($interviews as $interview): ?>
                            <tr>
                                <td>
                                    <strong><?php echo date('d-M-Y', strtotime($interview['interview_date'])); ?></strong><br>
                                    <small class="text-muted"><?php echo $interview['interview_time'] ? date('H:i', strtotime($interview['interview_time'])) : 'N/A'; ?></small>
                                </td>
                                <td>
                                    <span class="badge bg-info">
                                        <?php echo ucfirst(str_replace('_', ' ', $interview['interview_type'])); ?>
                                    </span><br>
                                    <small class="text-muted"><?php echo ucfirst($interview['status']); ?></small>
                                </td>
                                <td>
                                    <strong><?php echo htmlspecialchars($interview['interviewer_display_name']); ?></strong>
                                </td>
                                <td>
                                    <?php echo $interview['duration_minutes'] ? $interview['duration_minutes'] . ' min' : 'N/A'; ?>
                                </td>
                                <td>
                                    <span class="badge bg-<?php echo $interview['turn_up_status'] === 'turned_up' ? 'success' : 'danger'; ?>">
                                        <?php echo ucfirst(str_replace('_', ' ', $interview['turn_up_status'] ?? 'turned_up')); ?>
                                    </span>
                                    <?php if ($interview['absence_reason']): ?>
                                    <br><small class="text-muted"><?php echo htmlspecialchars($interview['absence_reason']); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="badge bg-<?php 
                                        echo match($interview['result']) {
                                            'selected' => 'success',
                                            'rejected', 'process_ended' => 'danger',
                                            'hold' => 'warning',
                                            'shortlisted' => 'info',
                                            'rescheduled' => 'secondary',
                                            default => 'primary'
                                        };
                                    ?>">
                                        <?php echo ucfirst(str_replace('_', ' ', $interview['result'])); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php if ($interview['technical_score'] || $interview['communication_score'] || $interview['overall_score']): ?>
                                        <?php if ($interview['technical_score']): ?>
                                        <span class="score-badge score-<?php echo $interview['technical_score'] >= 8 ? 'excellent' : ($interview['technical_score'] >= 6 ? 'good' : ($interview['technical_score'] >= 4 ? 'average' : 'poor')); ?>">
                                            Tech: <?php echo $interview['technical_score']; ?>/10
                                        </span><br>
                                        <?php endif; ?>
                                        <?php if ($interview['communication_score']): ?>
                                        <span class="score-badge score-<?php echo $interview['communication_score'] >= 8 ? 'excellent' : ($interview['communication_score'] >= 6 ? 'good' : ($interview['communication_score'] >= 4 ? 'average' : 'poor')); ?>">
                                            Comm: <?php echo $interview['communication_score']; ?>/10
                                        </span><br>
                                        <?php endif; ?>
                                        <?php if ($interview['overall_score']): ?>
                                        <span class="score-badge score-<?php echo $interview['overall_score'] >= 8 ? 'excellent' : ($interview['overall_score'] >= 6 ? 'good' : ($interview['overall_score'] >= 4 ? 'average' : 'poor')); ?>">
                                            Overall: <?php echo $interview['overall_score']; ?>/10
                                        </span>
                                        <?php endif; ?>
                                    <?php else: ?>
                                        <small class="text-muted">No scores</small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($interview['notes']): ?>
                                    <button class="btn btn-sm btn-outline-info" onclick="showNotes('<?php echo htmlspecialchars(addslashes($interview['notes'])); ?>')">
                                        <i class="fas fa-eye"></i> View
                                    </button>
                                    <?php else: ?>
                                    <small class="text-muted">No notes</small>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>

            <!-- Detailed Interview Feedback -->
            <?php if (!empty($feedback_records)): ?>
            <div class="info-card">
                <div class="section-header">
                    <h5 class="mb-0"><i class="fas fa-star me-2"></i>Interview Feedback & Scoring</h5>
                </div>
                <?php foreach ($feedback_records as $feedback): ?>
                <div class="border rounded p-3 mb-3">
                    <div class="row">
                        <div class="col-md-8">
                            <h6><i class="fas fa-user me-2"></i><?php echo htmlspecialchars($feedback['interviewer_name']); ?></h6>
                            <p class="text-muted mb-2">Feedback Date: <?php echo date('d-M-Y', strtotime($feedback['feedback_date'])); ?></p>
                        </div>
                        <div class="col-md-4 text-end">
                            <span class="badge bg-<?php echo $feedback['would_hire'] === 'yes' ? 'success' : ($feedback['would_hire'] === 'no' ? 'danger' : 'warning'); ?> fs-6">
                                Would Hire: <?php echo strtoupper($feedback['would_hire'] ?? 'N/A'); ?>
                            </span>
                        </div>
                    </div>
                    
                    <div class="row mb-3">
                        <div class="col-md-2">
                            <small class="text-muted">Technical Skills</small><br>
                            <span class="score-badge score-<?php echo ($feedback['technical_skills_score'] ?? 0) >= 8 ? 'excellent' : (($feedback['technical_skills_score'] ?? 0) >= 6 ? 'good' : (($feedback['technical_skills_score'] ?? 0) >= 4 ? 'average' : 'poor')); ?>">
                                <?php echo $feedback['technical_skills_score'] ?? 'N/A'; ?>/10
                            </span>
                        </div>
                        <div class="col-md-2">
                            <small class="text-muted">Communication</small><br>
                            <span class="score-badge score-<?php echo ($feedback['communication_score'] ?? 0) >= 8 ? 'excellent' : (($feedback['communication_score'] ?? 0) >= 6 ? 'good' : (($feedback['communication_score'] ?? 0) >= 4 ? 'average' : 'poor')); ?>">
                                <?php echo $feedback['communication_score'] ?? 'N/A'; ?>/10
                            </span>
                        </div>
                        <div class="col-md-2">
                            <small class="text-muted">Problem Solving</small><br>
                            <span class="score-badge score-<?php echo ($feedback['problem_solving_score'] ?? 0) >= 8 ? 'excellent' : (($feedback['problem_solving_score'] ?? 0) >= 6 ? 'good' : (($feedback['problem_solving_score'] ?? 0) >= 4 ? 'average' : 'poor')); ?>">
                                <?php echo $feedback['problem_solving_score'] ?? 'N/A'; ?>/10
                            </span>
                        </div>
                        <div class="col-md-2">
                            <small class="text-muted">Cultural Fit</small><br>
                            <span class="score-badge score-<?php echo ($feedback['cultural_fit_score'] ?? 0) >= 8 ? 'excellent' : (($feedback['cultural_fit_score'] ?? 0) >= 6 ? 'good' : (($feedback['cultural_fit_score'] ?? 0) >= 4 ? 'average' : 'poor')); ?>">
                                <?php echo $feedback['cultural_fit_score'] ?? 'N/A'; ?>/10
                            </span>
                        </div>
                        <div class="col-md-2">
                            <small class="text-muted">Leadership</small><br>
                            <span class="score-badge score-<?php echo ($feedback['leadership_score'] ?? 0) >= 8 ? 'excellent' : (($feedback['leadership_score'] ?? 0) >= 6 ? 'good' : (($feedback['leadership_score'] ?? 0) >= 4 ? 'average' : 'poor')); ?>">
                                <?php echo $feedback['leadership_score'] ?? 'N/A'; ?>/10
                            </span>
                        </div>
                        <div class="col-md-2">
                            <small class="text-muted">Overall</small><br>
                            <span class="score-badge score-<?php echo ($feedback['overall_score'] ?? 0) >= 8 ? 'excellent' : (($feedback['overall_score'] ?? 0) >= 6 ? 'good' : (($feedback['overall_score'] ?? 0) >= 4 ? 'average' : 'poor')); ?>">
                                <?php echo $feedback['overall_score'] ?? 'N/A'; ?>/10
                            </span>
                        </div>
                    </div>
                    
                    <?php if ($feedback['strengths']): ?>
                    <div class="mb-2">
                        <strong class="text-success"><i class="fas fa-plus-circle me-1"></i>Strengths:</strong>
                        <p class="mb-1"><?php echo nl2br(htmlspecialchars($feedback['strengths'])); ?></p>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($feedback['weaknesses']): ?>
                    <div class="mb-2">
                        <strong class="text-warning"><i class="fas fa-minus-circle me-1"></i>Areas for Improvement:</strong>
                        <p class="mb-1"><?php echo nl2br(htmlspecialchars($feedback['weaknesses'])); ?></p>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($feedback['detailed_feedback']): ?>
                    <div class="mb-2">
                        <strong><i class="fas fa-comment me-1"></i>Detailed Feedback:</strong>
                        <p class="mb-1"><?php echo nl2br(htmlspecialchars($feedback['detailed_feedback'])); ?></p>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($feedback['recommendations']): ?>
                    <div>
                        <strong class="text-info"><i class="fas fa-lightbulb me-1"></i>Recommendations:</strong>
                        <p class="mb-0"><?php echo nl2br(htmlspecialchars($feedback['recommendations'])); ?></p>
                    </div>
                    <?php endif; ?>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>

            <!-- Process Timeline -->
            <?php if (!empty($timeline)): ?>
            <div class="info-card">
                <div class="section-header">
                    <h5 class="mb-0"><i class="fas fa-timeline me-2"></i>Process Timeline</h5>
                </div>
                <div class="timeline">
                    <?php foreach ($timeline as $step): ?>
                    <div class="timeline-item">
                        <div class="d-flex justify-content-between align-items-start">
                            <div>
                                <h6 class="mb-1">
                                    <span class="badge bg-primary me-2"><?php echo $step['step_number']; ?></span>
                                    <?php echo htmlspecialchars($step['step_name']); ?>
                                </h6>
                                <p class="text-muted mb-2">
                                    <i class="fas fa-user me-1"></i><?php echo htmlspecialchars($step['performed_by_name'] ?? 'System'); ?>
                                    <span class="mx-2">|</span>
                                    <i class="fas fa-clock me-1"></i><?php echo date('d-M-Y H:i', strtotime($step['completed_at'] ?? $step['created_at'])); ?>
                                </p>
                                <?php if ($step['notes']): ?>
                                <p class="mb-0"><small><?php echo htmlspecialchars($step['notes']); ?></small></p>
                                <?php endif; ?>
                            </div>
                            <span class="badge bg-<?php echo $step['step_status'] === 'completed' ? 'success' : ($step['step_status'] === 'failed' ? 'danger' : 'warning'); ?>">
                                <?php echo ucfirst($step['step_status']); ?>
                            </span>
                        </div>
                        <?php if ($step['step_data']): ?>
                        <div class="mt-2">
                            <button class="btn btn-sm btn-outline-secondary" onclick="toggleStepData('step_<?php echo $step['id']; ?>')">
                                <i class="fas fa-info-circle me-1"></i>View Details
                            </button>
                            <div id="step_<?php echo $step['id']; ?>" class="mt-2" style="display: none;">
                                <pre class="bg-light p-2 rounded"><code><?php echo json_encode(json_decode($step['step_data']), JSON_PRETTY_PRINT); ?></code></pre>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>

            <!-- Joining Details -->
            <?php if ($joining_details): ?>
            <div class="info-card">
                <div class="section-header">
                    <h5 class="mb-0"><i class="fas fa-handshake me-2"></i>Joining Information</h5>
                </div>
                <div class="row">
                    <div class="col-md-6">
                        <table class="table table-borderless table-sm">
                            <tr>
                                <td><strong>Expected Joining:</strong></td>
                                <td><?php echo $joining_details['expected_joining_date'] ? date('d-M-Y', strtotime($joining_details['expected_joining_date'])) : 'N/A'; ?></td>
                            </tr>
                            <tr>
                                <td><strong>Actual Joining:</strong></td>
                                <td><?php echo $joining_details['actual_joining_date'] ? date('d-M-Y', strtotime($joining_details['actual_joining_date'])) : 'N/A'; ?></td>
                            </tr>
                            <tr>
                                <td><strong>Joining Status:</strong></td>
                                <td>
                                    <span class="badge bg-<?php echo $joining_details['joining_status'] === 'joined' ? 'success' : ($joining_details['joining_status'] === 'not_joined' ? 'danger' : 'warning'); ?>">
                                        <?php echo ucfirst(str_replace('_', ' ', $joining_details['joining_status'])); ?>
                                    </span>
                                </td>
                            </tr>
                            <tr>
                                <td><strong>Salary Offered:</strong></td>
                                <td><?php echo $joining_details['salary_offered'] ? '₹' . number_format($joining_details['salary_offered']) : 'N/A'; ?></td>
                            </tr>
                            <?php if ($joining_details['salary_negotiated']): ?>
                            <tr>
                                <td><strong>Salary Negotiated:</strong></td>
                                <td><?php echo '₹' . number_format($joining_details['salary_negotiated']); ?></td>
                            </tr>
                            <?php endif; ?>
                        </table>
                    </div>
                    <div class="col-md-6">
                        <table class="table table-borderless table-sm">
                            <tr>
                                <td><strong>Position:</strong></td>
                                <td><?php echo htmlspecialchars($joining_details['position'] ?? 'N/A'); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Department:</strong></td>
                                <td><?php echo htmlspecialchars($joining_details['department'] ?? 'N/A'); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Reporting Manager:</strong></td>
                                <td><?php echo htmlspecialchars($joining_details['reporting_manager'] ?? 'N/A'); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Work Location:</strong></td>
                                <td><?php echo htmlspecialchars($joining_details['work_location'] ?? 'N/A'); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Employment Type:</strong></td>
                                <td><?php echo ucfirst(str_replace('_', ' ', $joining_details['employment_type'] ?? 'N/A')); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Probation Period:</strong></td>
                                <td><?php echo $joining_details['probation_period_months'] ? $joining_details['probation_period_months'] . ' months' : 'N/A'; ?></td>
                            </tr>
                        </table>
                    </div>
                </div>
                
                <?php if ($joining_details['joining_notes'] || $joining_details['joining_feedback']): ?>
                <div class="row mt-3">
                    <div class="col-12">
                        <?php if ($joining_details['joining_notes']): ?>
                        <div class="mb-3">
                            <strong>Joining Instructions:</strong>
                            <p class="bg-light p-2 rounded"><?php echo nl2br(htmlspecialchars($joining_details['joining_notes'])); ?></p>
                        </div>
                        <?php endif; ?>
                        <?php if ($joining_details['joining_feedback']): ?>
                        <div class="mb-3">
                            <strong>Joining Feedback:</strong>
                            <p class="bg-light p-2 rounded"><?php echo nl2br(htmlspecialchars($joining_details['joining_feedback'])); ?></p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <!-- Reschedule History -->
            <?php if (!empty($reschedules)): ?>
            <div class="info-card">
                <div class="section-header">
                    <h5 class="mb-0"><i class="fas fa-calendar-alt me-2"></i>Reschedule History</h5>
                </div>
                <div class="table-responsive">
                    <table class="table table-sm data-table">
                        <thead>
                            <tr>
                                <th>Original Date</th>
                                <th>New Date</th>
                                <th>Reason</th>
                                <th>Requested By</th>
                                <th>Count</th>
                                <th>Details</th>
                                <th>Date</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($reschedules as $reschedule): ?>
                            <tr>
                                <td><?php echo date('d-M-Y', strtotime($reschedule['original_date'])); ?></td>
                                <td><?php echo date('d-M-Y', strtotime($reschedule['new_date'])); ?></td>
                                <td>
                                    <span class="badge bg-warning">
                                        <?php echo ucfirst(str_replace('_', ' ', $reschedule['reason'])); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php echo htmlspecialchars($reschedule['rescheduled_by_name']); ?><br>
                                    <small class="text-muted">(<?php echo ucfirst($reschedule['requested_by']); ?>)</small>
                                </td>
                                <td>
                                    <span class="badge bg-info">#<?php echo $reschedule['reschedule_count']; ?></span>
                                </td>
                                <td>
                                    <?php if ($reschedule['reason_details']): ?>
                                    <button class="btn btn-sm btn-outline-info" onclick="showNotes('<?php echo htmlspecialchars(addslashes($reschedule['reason_details'])); ?>')">
                                        <i class="fas fa-eye"></i> View
                                    </button>
                                    <?php else: ?>
                                    <small class="text-muted">No details</small>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo date('d-M-Y H:i', strtotime($reschedule['created_at'])); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>

            <!-- Status History -->
            <?php if (!empty($status_history)): ?>
            <div class="info-card">
                <div class="section-header">
                    <h5 class="mb-0"><i class="fas fa-history me-2"></i>Status Change History</h5>
                </div>
                <div class="table-responsive">
                    <table class="table table-sm data-table">
                        <thead>
                            <tr>
                                <th>Date & Time</th>
                                <th>Previous Status</th>
                                <th>New Status</th>
                                <th>Changed By</th>
                                <th>Reason</th>
                                <th>Additional Data</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($status_history as $history): ?>
                            <tr>
                                <td><?php echo date('d-M-Y H:i', strtotime($history['changed_at'])); ?></td>
                                <td>
                                    <span class="badge bg-secondary">
                                        <?php echo ucfirst(str_replace('_', ' ', $history['previous_status'] ?? 'N/A')); ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="badge bg-primary">
                                        <?php echo ucfirst(str_replace('_', ' ', $history['new_status'])); ?>
                                    </span>
                                </td>
                                <td><?php echo htmlspecialchars($history['changed_by_name'] ?? 'System'); ?></td>
                                <td>
                                    <small><?php echo htmlspecialchars(substr($history['reason'] ?? '', 0, 50)) . (strlen($history['reason'] ?? '') > 50 ? '...' : ''); ?></small>
                                </td>
                                <td>
                                    <?php if ($history['additional_data']): ?>
                                    <button class="btn btn-sm btn-outline-secondary" onclick="showAdditionalData('<?php echo htmlspecialchars(addslashes($history['additional_data'])); ?>')">
                                        <i class="fas fa-database"></i> View
                                    </button>
                                    <?php else: ?>
                                    <small class="text-muted">None</small>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>

            <!-- Offers -->
            <?php if (!empty($offers)): ?>
            <div class="info-card">
                <div class="section-header">
                    <h5 class="mb-0"><i class="fas fa-file-contract me-2"></i>Offer Details</h5>
                </div>
                <?php foreach ($offers as $offer): ?>
                <div class="border rounded p-3 mb-3">
                    <div class="row">
                        <div class="col-md-8">
                            <h6>
                                <?php echo ucfirst($offer['offer_type']); ?> Offer - 
                                <span class="badge bg-<?php echo $offer['offer_status'] === 'accepted' ? 'success' : ($offer['offer_status'] === 'declined' ? 'danger' : 'warning'); ?>">
                                    <?php echo ucfirst($offer['offer_status']); ?>
                                </span>
                            </h6>
                            <p class="mb-1"><strong>Position:</strong> <?php echo htmlspecialchars($offer['position']); ?></p>
                            <p class="mb-1"><strong>Department:</strong> <?php echo htmlspecialchars($offer['department']); ?></p>
                        </div>
                        <div class="col-md-4 text-end">
                            <h5 class="text-success">₹<?php echo number_format($offer['salary_offered']); ?></h5>
                            <?php if ($offer['salary_negotiated']): ?>
                            <p class="mb-0"><small>Negotiated: ₹<?php echo number_format($offer['salary_negotiated']); ?></small></p>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="row mt-2">
                        <div class="col-md-6">
                            <small class="text-muted">Created by:</small> <?php echo htmlspecialchars($offer['created_by_name']); ?><br>
                            <small class="text-muted">Created on:</small> <?php echo date('d-M-Y', strtotime($offer['created_at'])); ?>
                        </div>
                        <div class="col-md-6">
                            <?php if ($offer['offer_sent_date']): ?>
                            <small class="text-muted">Sent on:</small> <?php echo date('d-M-Y', strtotime($offer['offer_sent_date'])); ?><br>
                            <?php endif; ?>
                            <?php if ($offer['offer_response_date']): ?>
                            <small class="text-muted">Response on:</small> <?php echo date('d-M-Y', strtotime($offer['offer_response_date'])); ?>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <?php if ($offer['benefits_offered'] || $offer['special_conditions']): ?>
                    <div class="mt-3">
                        <?php if ($offer['benefits_offered']): ?>
                        <div class="mb-2">
                            <strong>Benefits:</strong>
                            <p class="bg-light p-2 rounded mb-1"><?php echo nl2br(htmlspecialchars($offer['benefits_offered'])); ?></p>
                        </div>
                        <?php endif; ?>
                        <?php if ($offer['special_conditions']): ?>
                        <div>
                            <strong>Special Conditions:</strong>
                            <p class="bg-light p-2 rounded mb-0"><?php echo nl2br(htmlspecialchars($offer['special_conditions'])); ?></p>
                        </div>
                        <?php endif; ?>
                    </div>
                    <?php endif; ?>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>

            <!-- Action Buttons -->
            <div class="text-center mt-4 no-print">
                <button type="button" class="btn btn-secondary me-2" onclick="window.history.back()">
                    <i class="fas fa-arrow-left me-1"></i>Back
                </button>
                <button type="button" class="btn btn-primary me-2" onclick="editCandidate()">
                    <i class="fas fa-edit me-1"></i>Edit Candidate
                </button>
                <button type="button" class="btn btn-info me-2" onclick="window.print()">
                    <i class="fas fa-print me-1"></i>Print Report
                </button>
                <?php if (in_array($candidate['status'], ['pending', 'shortlisted', 'rescheduled'])): ?>
                <button type="button" class="btn btn-success" onclick="startInterviewProcess()">
                    <i class="fas fa-play me-1"></i>Continue Interview Process
                </button>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Notes Modal -->
    <div class="modal fade" id="notesModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Notes/Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <pre id="notesContent" class="bg-light p-3 rounded"></pre>
                </div>
            </div>
        </div>
    </div>

    <!-- Additional Data Modal -->
    <div class="modal fade" id="dataModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Additional Data</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <pre id="dataContent" class="bg-light p-3 rounded"></pre>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function showNotes(notes) {
            document.getElementById('notesContent').textContent = notes;
            new bootstrap.Modal(document.getElementById('notesModal')).show();
        }

        function showAdditionalData(data) {
            try {
                const parsed = JSON.parse(data);
                document.getElementById('dataContent').textContent = JSON.stringify(parsed, null, 2);
            } catch (e) {
                document.getElementById('dataContent').textContent = data;
            }
            new bootstrap.Modal(document.getElementById('dataModal')).show();
        }

        function toggleStepData(elementId) {
            const element = document.getElementById(elementId);
            element.style.display = element.style.display === 'none' ? 'block' : 'none';
        }

        function editCandidate() {
            window.location.href = `edit-candidate.html?edit=<?php echo $candidate['candidate_id']; ?>`;
        }

        function startInterviewProcess() {
            // Open interview process modal or redirect
            window.location.href = `search-candidates.php?interview=<?php echo $candidate['candidate_id']; ?>`;
        }

        function logout() {
            if (confirm('Are you sure you want to logout?')) {
                window.location.href = 'logout.php';
            }
        }

        // Print optimization
        window.addEventListener('beforeprint', function() {
            document.querySelectorAll('.btn').forEach(btn => btn.style.display = 'none');
        });

        window.addEventListener('afterprint', function() {
            document.querySelectorAll('.btn').forEach(btn => btn.style.display = '');
        });
    </script>
</body>
</html>
