<?php
include_once '../config/database.php';
include_once '../classes/Candidate.php';
include_once '../classes/Interview.php';

session_start();
if (!isset($_SESSION['user_id'])) {
    exit('Unauthorized');
}

$database = new Database();
$db = $database->getConnection();
$candidate = new Candidate($db);
$interview = new Interview($db);

$candidate_id = $_GET['candidate_id'] ?? '';
$candidate->candidate_id = $candidate_id;

if (!$candidate->readOne()) {
    exit('Candidate not found');
}

// Get interview history
$interview_stmt = $interview->readByCandidate($candidate->id);
$interviews = $interview_stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<div class="candidate-details">
    <div class="row">
        <div class="col-md-6">
            <h5><i class="fas fa-user me-2"></i>Personal Information</h5>
            <table class="table table-borderless table-sm">
                <tr>
                    <td><strong>Candidate ID:</strong></td>
                    <td><?php echo htmlspecialchars($candidate->candidate_id); ?></td>
                </tr>
                <tr>
                    <td><strong>Name:</strong></td>
                    <td><?php echo htmlspecialchars($candidate->name); ?></td>
                </tr>
                <tr>
                    <td><strong>Email:</strong></td>
                    <td><a href="mailto:<?php echo $candidate->email; ?>"><?php echo htmlspecialchars($candidate->email); ?></a></td>
                </tr>
                <tr>
                    <td><strong>Phone:</strong></td>
                    <td><a href="tel:<?php echo $candidate->phone; ?>"><?php echo htmlspecialchars($candidate->phone); ?></a></td>
                </tr>
                <tr>
                    <td><strong>Location:</strong></td>
                    <td><?php echo htmlspecialchars($candidate->location); ?></a></td>
                </tr>
                <tr>
                    <td><strong>Team Name:</strong></td>
                    <td><?php echo htmlspecialchars($candidate->team_name); ?></a></td>
                </tr>
            </table>
        </div>
        <div class="col-md-6">
            <h5><i class="fas fa-briefcase me-2"></i>Job Information</h5>
            <table class="table table-borderless table-sm">
                <tr>
                    <td><strong>Source:</strong></td>
                    <td><?php echo htmlspecialchars($candidate->source_name ?? 'N/A'); ?></td>
                </tr>
                <tr>
                    <td><strong>Profile:</strong></td>
                    <td><?php echo htmlspecialchars($candidate->profile_name ?? 'N/A'); ?></td>
                </tr>
                <tr>
                    <td><strong>Lineup Date:</strong></td>
                    <td><?php echo date('d-M-Y', strtotime($candidate->lineup_date)); ?></td>
                </tr>
                <tr>
                    <td><strong>Added Date:</strong></td>
                    <td><?php echo date('d-M-Y', strtotime($candidate->added_date)); ?></td>
                </tr>
                <tr>
                    <td><strong>Status:</strong></td>
                    <td>
                        <span class="badge bg-<?php 
                            echo match($candidate->status) {
                                'selected', 'joined' => 'success',
                                'rejected', 'process_ended' => 'danger',
                                'hold' => 'warning',
                                'shortlisted' => 'info',
                                'rescheduled' => 'secondary',
                                default => 'primary'
                            };
                        ?>">
                            <?php echo strtoupper(str_replace('_', ' ', $candidate->status)); ?>
                        </span>
                    </td>
                </tr>
            </table>
        </div>
    </div>

    <?php if (!empty($interviews)): ?>
    <hr>
    <h5><i class="fas fa-history me-2"></i>Interview History</h5>
    <div class="table-responsive">
        <table class="table table-sm table-striped">
            <thead class="table-dark">
                <tr>
                    <th>Date</th>
                    <th>Type</th>
                    <th>Duration</th>
                    <th>Status</th>
                    <th>Result</th>
                    <th>Interviewer</th>
                    <th>Notes</th>
                    <th>Next Interview</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($interviews as $int): ?>
                <tr>
                    <td><?php echo date('d-M-Y', strtotime($int['interview_date'])); ?></td>
                    <td>
                        <span class="badge bg-secondary">
                            <?php echo ucfirst(str_replace('_', ' ', $int['interview_type'])); ?>
                        </span>
                    </td>
                    <td><?php echo $int['duration_minutes'] ? $int['duration_minutes'] . ' min' : 'N/A'; ?></td>
                    <td>
                        <span class="badge bg-<?php echo $int['status'] === 'completed' ? 'success' : 'warning'; ?>">
                            <?php echo ucfirst($int['status']); ?>
                        </span>
                    </td>
                    <td>
                        <span class="badge bg-<?php 
                            echo match($int['result']) {
                                'selected' => 'success',
                                'rejected', 'process_ended' => 'danger',
                                'hold' => 'warning',
                                'shortlisted' => 'info',
                                default => 'secondary'
                            };
                        ?>">
                            <?php echo ucfirst(str_replace('_', ' ', $int['result'])); ?>
                        </span>
                    </td>
                    <td><?php echo htmlspecialchars($int['interviewer_name'] ?? 'N/A'); ?></td>
                    <td>
                        <small><?php echo htmlspecialchars(substr($int['notes'] ?? '', 0, 50)) . (strlen($int['notes'] ?? '') > 50 ? '...' : ''); ?></small>
                    </td>
                    <td>
                        <?php if ($int['next_interview_date']): ?>
                            <small>
                                <?php echo date('d-M-Y', strtotime($int['next_interview_date'])); ?><br>
                                <em><?php echo ucfirst($int['next_interview_type'] ?? ''); ?></em>
                            </small>
                        <?php else: ?>
                            <small class="text-muted">N/A</small>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <?php else: ?>
    <hr>
    <div class="alert alert-info">
        <i class="fas fa-info-circle me-2"></i>No interview history available for this candidate.
    </div>
    <?php endif; ?>

    <?php
    // Check for joining details if candidate is selected or joined
    if (in_array($candidate->status, ['selected', 'joined', 'selected_not_joined'])) {
        $joining_query = "SELECT * FROM joining_details WHERE candidate_id = ? ORDER BY created_at DESC LIMIT 1";
        $joining_stmt = $db->prepare($joining_query);
        $joining_stmt->execute([$candidate->id]);
        $joining_details = $joining_stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($joining_details):
    ?>
    <hr>
    <h5><i class="fas fa-handshake me-2"></i>Joining Details</h5>
    <div class="row">
        <div class="col-md-6">
            <table class="table table-borderless table-sm">
                <tr>
                    <td><strong>Expected Joining:</strong></td>
                    <td><?php echo $joining_details['expected_joining_date'] ? date('d-M-Y', strtotime($joining_details['expected_joining_date'])) : 'N/A'; ?></td>
                </tr>
                <tr>
                    <td><strong>Actual Joining:</strong></td>
                    <td><?php echo $joining_details['actual_joining_date'] ? date('d-M-Y', strtotime($joining_details['actual_joining_date'])) : 'N/A'; ?></td>
                </tr>
                <tr>
                    <td><strong>Joining Status:</strong></td>
                    <td>
                        <span class="badge bg-<?php echo $joining_details['joining_status'] === 'joined' ? 'success' : 'warning'; ?>">
                            <?php echo ucfirst(str_replace('_', ' ', $joining_details['joining_status'])); ?>
                        </span>
                    </td>
                </tr>
                <tr>
                    <td><strong>Salary Offered:</strong></td>
                    <td><?php echo $joining_details['salary_offered'] ? '₹' . number_format($joining_details['salary_offered']) : 'N/A'; ?></td>
                </tr>
            </table>
        </div>
        <div class="col-md-6">
            <table class="table table-borderless table-sm">
                <tr>
                    <td><strong>Department:</strong></td>
                    <td><?php echo htmlspecialchars($joining_details['department'] ?? 'N/A'); ?></td>
                </tr>
                <tr>
                    <td><strong>Position:</strong></td>
                    <td><?php echo htmlspecialchars($joining_details['position'] ?? 'N/A'); ?></td>
                </tr>
                <tr>
                    <td><strong>Reporting Manager:</strong></td>
                    <td><?php echo htmlspecialchars($joining_details['reporting_manager'] ?? 'N/A'); ?></td>
                </tr>
                <tr>
                    <td><strong>Work Location:</strong></td>
                    <td><?php echo htmlspecialchars($joining_details['work_location'] ?? 'N/A'); ?></td>
                </tr>
            </table>
        </div>
    </div>
    <?php if ($joining_details['joining_notes'] || $joining_details['joining_feedback']): ?>
    <div class="row">
        <div class="col-12">
            <?php if ($joining_details['joining_notes']): ?>
            <p><strong>Joining Notes:</strong><br>
            <small><?php echo nl2br(htmlspecialchars($joining_details['joining_notes'])); ?></small></p>
            <?php endif; ?>
            <?php if ($joining_details['joining_feedback']): ?>
            <p><strong>Joining Feedback:</strong><br>
            <small><?php echo nl2br(htmlspecialchars($joining_details['joining_feedback'])); ?></small></p>
            <?php endif; ?>
        </div>
    </div>
    <?php endif; ?>
    <?php endif; } ?>

    <?php
    // Check for reschedule history
    $reschedule_query = "SELECT * FROM reschedules WHERE candidate_id = ? ORDER BY created_at DESC";
    $reschedule_stmt = $db->prepare($reschedule_query);
    $reschedule_stmt->execute([$candidate->id]);
    $reschedules = $reschedule_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (!empty($reschedules)):
    ?>
    <hr>
    <h5><i class="fas fa-calendar-alt me-2"></i>Reschedule History</h5>
    <div class="table-responsive">
        <table class="table table-sm table-striped">
            <thead class="table-dark">
                <tr>
                    <th>Original Date</th>
                    <th>New Date</th>
                    <th>Reason</th>
                    <th>Details</th>
                    <th>Rescheduled On</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($reschedules as $reschedule): ?>
                <tr>
                    <td><?php echo date('d-M-Y', strtotime($reschedule['original_date'])); ?></td>
                    <td><?php echo date('d-M-Y', strtotime($reschedule['new_date'])); ?></td>
                    <td><span class="badge bg-warning"><?php echo $reschedule['reason']; ?></span></td>
                    <td><small><?php echo htmlspecialchars($reschedule['reason_details'] ?? 'N/A'); ?></small></td>
                    <td><?php echo date('d-M-Y H:i', strtotime($reschedule['created_at'])); ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <?php endif; ?>

    <div class="text-center mt-4">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
            <i class="fas fa-times me-1"></i>Close
        </button>
        <button type="button" class="btn btn-primary" onclick="editCandidate('<?php echo $candidate->candidate_id; ?>')">
            <i class="fas fa-edit me-1"></i>Edit Details
        </button>
        <button type="button" class="btn btn-info" onclick="printCandidate('<?php echo $candidate->candidate_id; ?>')">
            <i class="fas fa-print me-1"></i>Print Details
        </button>
    </div>
</div>

<script>
function printCandidate(candidateId) {
    // Create a new window for printing
    var printWindow = window.open('', '_blank');
    var candidateDetails = document.querySelector('.candidate-details').cloneNode(true);
    
    // Remove action buttons from print version
    var actionButtons = candidateDetails.querySelector('.text-center.mt-4');
    if (actionButtons) {
        actionButtons.remove();
    }
    
    printWindow.document.write(`
        <!DOCTYPE html>
        <html>
        <head>
            <title>Candidate Details - ${candidateId}</title>
            <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
            <style>
                body { font-size: 12px; }
                .badge { font-size: 10px; }
                @media print {
                    .no-print { display: none; }
                }
            </style>
        </head>
        <body>
            <div class="container-fluid">
                <h3 class="text-center mb-4">Candidate Details Report</h3>
                ${candidateDetails.outerHTML}
            </div>
        </body>
        </html>
    `);
    
    printWindow.document.close();
    printWindow.focus();
    setTimeout(() => {
        printWindow.print();
        printWindow.close();
    }, 250);
}
</script>
